(ns hara.io.scheduler.tab
  (:require [hara.string :as string]
            [hara.core.base.error :as error]
            [hara.core.base.util :as primitive]
            [hara.time :as time]))

(defonce +schedule-elements+ [:second :minute :hour :day-of-week :day :month :year])

 ;; There are 2 different representations of schedular tab data:
 ;;   string: (for humans)        "   *       2,4      2-9         /8      ...  "
 ;;    array: (for efficiency)    [ (-*)   [2 4 6]   (-- 2 9)   (-- 8)     ...  ]
 ;;
 ;;            :tab-str                   :tab-arr
 ;;           +---------+                 +---------+
 ;;           |         |                 |         |
 ;;           |         |                 |         |
 ;;           | string  |    -----------> |  array  |
 ;;           |         |    parse-tab    |         |
 ;;           |         |                 |         |
 ;;           +---------+                 +---------+
 ;;            for human                    used in
 ;;            use to add                  cronj-loop
 ;;            tasks

;; Methods for type conversion
(defn- to-long [x] (Long/parseLong x))

;; Array Representation
(defn *-
  "takes a string and returns something
   (*-) => :*
 
   (map (*- 2) (range 60))
   => (map even? (range 60))
 
   "
  {:added "3.0"}
  ([]      :*)
  ([s]     (fn [v] (zero? (mod v s))))
  ([a b]   (fn [v] (and (>= v a) (<= v b))))
  ([a b s] (fn [v] (and (>= v a)
                        (<= v b)
                        (zero? (mod (- v a) s))))))

;; String to Array Methods
(defn parse-tab-element
  "make a tab element into a array element
 
   (parse-tab-element \"1\") => 1
 
   (parse-tab-element \"*\") => :*
 
   (parse-tab-element \"1-5\") => fn?"
  {:added "3.0"}
  [^String es]
  (cond (= es "*") :*
        (re-find #"^\d+$" es) (to-long es)
        (re-find #"^/\d+$" es) (*- (to-long (.substring es 1)))
        (re-find #"^\d+-\d+$" es)
        (apply *-
               (sort (map to-long (string/split es #"-"))))
        (re-find #"^\d+-\d+/\d$" es)
        (apply *-
               (map to-long (string/split es #"[-/]")))
        :else (throw (ex-info "Input is not in the right format." {:input es}))))

(defn parse-tab-group
  "makes a tab group from a string separated by commas
 
   (parse-tab-group \"1,3,15\")
   => '(1 3 15)"
  {:added "3.0"}
  [s]
  (let [e-toks (re-seq #"[^,]+" s)]
    (map parse-tab-element e-toks)))

(defn parse-tab
  "takes a string and creates matches
 
   (parse-tab \"* * * * * * *\")
   => '[(:*) (:*) (:*) (:*) (:*) (:*) (:*)]
 
   (parse-tab \"* * * * * *\")
   => '[(0) (:*) (:*) (:*) (:*) (:*) (:*)]
 
   (parse-tab \"* * * * *\")
   => (throws Exception)
 
   "
  {:added "3.0"}
  [s]
  (let [c-toks (re-seq #"[^\s]+" s)
        len-c (count c-toks)
        sch-c  (count +schedule-elements+)]
    (cond (= sch-c len-c) (map parse-tab-group c-toks)
          (= (dec sch-c) len-c) (map parse-tab-group (cons "0" c-toks))
          :else
          (throw (ex-info "The schedule does not have the correct number of elements."
                          {:schedule s})))))

(defn valid-tab?
  "checks if the tab is valid
 
   (valid-tab? \"*\") => nil
 
   (valid-tab? \"* * * * * * *\") => true"
  {:added "3.0"}
  [s]
  (error/suppress (if (parse-tab s) true)))

;; dt-arr methods
(defn to-time-array
  "takes a time element and returns an array representation
 
   (to-time-array #inst \"1970-01-01T00:00:00.000-00:00\" \"UTC\")
   => [0 0 0 4 1 1 1970]
 
   (to-time-array #inst \"1970-01-01T00:00:00.000-00:00\" \"GMT-10\")
   => [0 0 14 3 31 12 1969]"
  {:added "3.0"}
  ([t] (to-time-array t (time/local-timezone)))
  ([t tz]
   (time/to-vector t {:timezone tz} +schedule-elements+)))

(defn match-element?
  "takes an element of the array and compares with a single matcher
 
   (match-element? 1 :*)
   => true
 
   (match-element? 1 [2 3 4])
   => false
 
   "
  {:added "3.0"}
  [dt-e tab-e]
  (or (cond (= tab-e :*) true
            (= tab-e dt-e) true
            (fn? tab-e) (tab-e dt-e)
            (sequential? tab-e) (some #(match-element? dt-e %) tab-e))
      false))

(defn match-array?
  "takes an array representation for match comparison
 
   (match-array? [30 14 0 4 26 7 2012]
                 [(*- 0 60 5) (*-) (*-) (*-) (*-) (*-) (*-)])
   => true
 
   (match-array? [31 14 0 4 26 7 2012]
                 [(*- 0 60 5) (*-) (*-) (*-) (*-) (*-) (*-)])
   => false"
  {:added "3.0"}
  [dt-array tab-array]
  (every? true?
          (map match-element? dt-array tab-array)))

(def nil-array (vec (repeat 7 primitive/F)))
