(ns hara.io.file.writer
  (:require [hara.io.file.charset :as charset]
            [hara.io.file.option :as option]
            [hara.io.file.path :as path])
  (:import (java.io BufferedWriter CharArrayWriter FileWriter FilterWriter
                    OutputStream OutputStreamWriter PipedWriter PrintWriter StringWriter Writer)
           (java.nio.file Path Files OpenOption)))

(defmulti -writer
  "creates a writer for a given input
 
   (doto (-writer :buffered {:path \"hello.txt\"})
     (.write \"Hello\" 0 4)
     (.write \"World\" 0 4)
     (.close))
 
   (slurp \"hello.txt\") => \"HellWorl\"
 
   "
  {:added "3.0"}
  (fn [type opts] type))

(defmethod -writer :buffered
  [_ {:keys [^Path path charset] :as opts}]
  (Files/newBufferedWriter (path/path path)
                           (charset/charset (or charset
                                                (charset/charset-default)))
                           (->> [:create]
                                (or (:options opts))
                                (mapv option/+open-options+)
                                (into-array OpenOption))))

(defmethod -writer :char-array
  [_ {:keys [size] :as opts}]
  (CharArrayWriter. (or size 0)))

(defmethod -writer :file
  [_ {:keys [^Path path ^Boolean append] :as opts :or {append false}}]
  (let [file (.toFile (path/path path))]
    (FileWriter. file append)))

(defmethod -writer :output-stream
  [_ {:keys [^OutputStream stream charset] :as opts}]
  (OutputStreamWriter. stream (charset/charset (or charset
                                                   (charset/charset-default)))))

(defmethod -writer :piped
  [_ {:keys [reader] :as opts}]
  (PipedWriter. reader))

(defmethod -writer :print
  [_ {:keys [^OutputStream out]}]
  (PrintWriter. out))

(defmethod -writer :string
  [_ {:keys [size] :as opts}]
  (StringWriter. size))

(defn writer-types
  "returns the types of writers
 
   (writer-types)
   => (contains [:buffered :char-array :file
                 :output-stream :piped :print :string]
                :in-any-order)"
  {:added "3.0"}
  []
  (keys (.getMethodTable ^clojure.lang.MultiFn -writer)))

(defn writer
  "creates a writer given options
 
   (doto (writer :buffered {:path \"hello.txt\"})
     (.write \"Hello\" 0 4)
     (.write \"World\" 0 4)
     (.close))
 
   (slurp \"hello.txt\") => \"HellWorl\"
 
   "
  {:added "3.0"}
  ([opts]
   (writer :buffered opts))
  ([type opts]
   (-writer type opts)))
