(ns hara.deploy.maven
  (:require [hara.data :as data]
            [hara.deploy.maven.command :as command]
            [hara.deploy.maven.package :as package]
            [hara.deploy.linkage :as linkage]
            [hara.deploy.task :as deploy.task]
            [hara.io.project :as project]
            [hara.function :refer [definvoke]]
            [hara.function.task :as task]
            [hara.lib.aether :as aether]))

(defmethod task/task-defaults :deploy.maven
  [_]
  (merge deploy.task/+main+
         {:item      {:list     (fn [lookup _] (vec (sort (keys lookup))))
                      :display  (fn [data] (vec (sort (map :extension data))))}
          :result    {:keys    {:artifacts (fn [data]
                                             (let [{:keys [version group artifact]} (first data)
                                                   extensions (sort (map (comp str :extension) data))]
                                               [group version extensions]))}
                      :columns [{:key    :key
                                 :align  :left}
                                {:key    :artifacts
                                 :align  :left
                                 :length 60
                                 :color  #{:bold}}
                                {:key    :time
                                 :align  :left
                                 :length 10
                                 :format "%d ms"
                                 :color  #{:bold}}]}
          :summary  {:aggregate {}}}))

(defmethod task/task-defaults :deploy.linkage
  [_]
  (merge deploy.task/+main+
         {:item      {:list    (fn [lookup _] (vec (sort (keys lookup))))}
          :result    {:columns [{:key    :key
                                 :align  :left}
                                {:key    :time
                                 :align  :left
                                 :length 10
                                 :format "%d ms"
                                 :color  #{:bold}}]}
          :summary  {:aggregate {}}}))

(defmethod task/task-defaults :deploy.package
  [_]
  (merge deploy.task/+main+
         {:item      {:list     (fn [lookup _] (vec (sort (keys lookup))))
                      :display  (juxt (comp count :files) :pom)}
          :result    {:keys    {:jar :jar
                                :pom :pom
                                :packaged (comp count :files)}
                      :columns [{:key    :key
                                 :align  :left}
                                {:key    :packaged
                                 :align  :left
                                 :format "(%d)"
                                 :length 10
                                 :color  #{:bold}}
                                {:key    :jar
                                 :align  :left
                                 :length 40
                                 :color  #{:bold}}
                                {:key    :time
                                 :align  :left
                                 :length 10
                                 :format "%d ms"
                                 :color  #{:bold}}]}
          :summary  {:aggregate {:packaged   [:packaged + 0]}}}))

(definvoke linkage
  "creates linkages for project
 
   (linkage :all {:print {:item false :result false :summary false}})"
  {:added "3.0"}
  [:task {:template :deploy.linkage
          :params {:title "CREATES ALL LINKAGE FILES"}
          :main {:fn #'package/linkage}}])

(definvoke package
  "packages files in the interim directory
 
   (package '[hara]
            {:tag :public
            :print {:item false :result false :summary false}})"
  {:added "3.0"}
  [:task {:template :deploy.package
          :params {:title "PACKAGE INTERIM FILES"}
          :main {:fn #'package/package}}])

(definvoke clean
  "cleans the interim directory of packages
 
   (clean :all {:print {:item false :result false :summary false}})"
  {:added "3.0"}
  [:task {:template :deploy.linkage
          :params {:title "CLEAN ALL INTERIM FILES"}
          :main {:fn #'command/clean}}])

(definvoke install
  "installs packages to the local `.m2` repository
 
   (install '[hara] {:tag :public})"
  {:added "3.0"}
  [:task {:template :deploy.maven
            :params {:title "INSTALL PACKAGES"}
          :main {:fn #'command/install}}])

(definvoke install-secure
  "installs signed packages to the local `.m2` repository
 
   (install-secure '[hara] {:tag :public})"
  {:added "3.0"}
  [:task {:template :deploy.maven
          :params {:title "INSTALL SIGNED PACKAGES"}
          :main {:fn #'command/install-secure}}])

(definvoke deploy
  "deploys packages to a maven repository
 
   (deploy '[hara] {:tag :public})"
  {:added "3.0"}
  [:task {:template :deploy.maven
          :params {:title "DEPLOY PACKAGES"}
          :main {:fn #'command/deploy}}])
