(ns hara.deploy.linkage
  (:require [clojure.set :as set]
            [hara.core.base.sort :as sort]
            [hara.data.base.map :as map]
            [hara.deploy.linkage.common :as common]
            [hara.deploy.linkage.clj]
            [hara.deploy.linkage.cljs]
            [hara.deploy.linkage.java]
            [hara.io.file :as fs]
            [hara.io.project :as project]
            [hara.module.artifact :as artifact]
            [hara.module.deps :as deps]))

;; Linkage takes
;; - read, where an .edn file consisting of 
;; - collect
;;   - entries
;;   - linkages
;;   - internal deps
;;   - external deps
;;   - transfers

(def +default-packages-file+  "config/packages.edn")

(def +suffix-types+ {:clj  ".clj$"
                     :cljs ".cljs$"
                     :cljc ".cljc$"})

(defn file-linkage
  "returns the exports and imports of a given file
   
   (file-linkage \"src/hara/deploy/linkage/common.clj\")
   => '{:exports #{[:class hara.deploy.linkage.common.FileInfo]
                   [:clj hara.deploy.linkage.common]},
        :imports #{[:clj hara.io.file]
                   [:clj hara.function]}}"
  {:added "3.0"}
  ([path]
  (common/file-linkage-fn path
                          (-> (fs/path path)
                              (fs/attributes)
                              (:last-modified-time)))))

(defn read-packages
  "reads in a list of packages to 
 
   (-> (read-packages {:file \"config/packages.edn\"})
       (get 'hara/core.version))
   => (contains {:description string?
                 :include '[[hara.core.version :complete]],
                 :name 'hara/core.version})"
  {:added "3.0"}
  ([]
   (read-packages {:root "."
                   :file +default-packages-file+}))
  ([{:keys [root file] :as input}]
   (let [path   (fs/path root file)]
     (if (fs/exists? path)
     (->> (read-string (slurp path))
          (map/map-entries (fn [[k entry]]
                             [k (assoc entry :name k)])))
     (throw (ex-info "Path does not exist" {:path path
                                            :input input}))))))

(defn create-file-lookups
  "creates file-lookups for clj, cljs and cljc files
 
   (-> (create-file-lookups (project/project))
       (get-in [:clj 'hara.core.version]))
   => (str (fs/path \"src/hara/core/version.clj\"))"
  {:added "3.0"}
  ([project]
   (map/map-vals (fn [suffix]
                   (project/all-files (:source-paths project)
                                      {:include [suffix]}
                                      project))
                 +suffix-types+)))

(defn collect-entries-single
  "collects all namespaces for given lookup and package
   
   
   (collect-entries-single (get -packages- 'hara/core.version)
                           (:clj -lookups-))
   => '(hara.core.version)"
  {:added "3.0"}
  ([package lookup]
   (let [nsps (keys lookup)]
     (mapcat (fn [[ns type]]
               (case type
                 :base      (filter (fn [sym] (or (= sym ns)
                                                  (.startsWith (str sym)
                                                               (str ns ".base."))))
                                    nsps)
                 :complete  (filter (fn [sym] (or (= sym ns)
                                                  (.startsWith (str sym)
                                                               (str ns "."))))
                                    nsps)
                 (throw (ex-info "Not supported." {:type type
                                                   :options [:base :complete]}))))
             (:include package)))))

(defn collect-entries
  "collects all entries given packages and lookups
   
   (-> (collect-entries -packages- -lookups-)
       (get-in '[hara/core.version :entries]))
   => '#{[:clj hara.core.version]}"
  {:added "3.0"}
  ([packages lookups]
   (map/map-vals (fn [pkg]
                   (->> lookups
                        (mapcat (fn [[suffix lookup]]
                                  (->> (collect-entries-single pkg lookup)
                                       (map (partial vector suffix)))))
                        (set)
                        (assoc pkg :entries)))
                 packages)))

(defn overlapped-entries-single
  "finds any overlaps between entries
   
   (overlapped-entries-single '{:name a
                                :entries #{[:clj hara.1]}}
                              '[{:name b
                                 :entries #{[:clj hara.1] [:clj hara.2]}}])
   => '([#{a b} #{[:clj hara.1]}])"
  {:added "3.0"}
  [x heap]
  (keep (fn [{:keys [name entries]}]
          (let [ol (set/intersection (:entries x)
                                     entries)]
            (if (seq ol)
              [#{name (:name x)} ol])))
        heap))

(defn overlapped-entries
  "finds any overlapped entries for given map
 
   (overlapped-entries '{a {:name a
                            :entries #{[:clj hara.1]}}
                         b {:name b
                            :entries #{[:clj hara.1] [:clj hara.2]}}})
   => '([#{a b} #{[:clj hara.1]}])"
  {:added "3.0"}
  [packages]
  (loop [[x & rest] (vals packages)
         heap     []
         overlaps []]
    (cond (nil? x)
          overlaps
          
          :else
          (let [ols (overlapped-entries-single x heap)]
            (recur rest
                   (conj heap x)
                   (concat overlaps ols))))))

(defn missing-entries
  "finds missing entries given packages and lookup
 
   (missing-entries '{b {:name b
                         :entries #{[:clj hara.1] [:clj hara.2]}}}
                    '{:clj {hara.1 \"\"
                            hara.2 \"\"
                            hara.3 \"\"}})
   => '{:clj {hara.3 \"\"}}"
  {:added "3.0"}
  [packages lookups]
  (reduce (fn [lookups {:keys [entries]}]
            (reduce (fn [lookups entry]
                      (map/dissoc-in lookups entry))
                    lookups
                    entries))
          (reduce-kv (fn [out k v]
                       (if (empty? v) out (assoc out k v)))
                     {}
                     lookups)
          (vals packages)))

(defn collect-external-deps
  "collects dependencies from the local system
 
   (collect-external-deps '{:a {:dependencies [org.clojure/clojure]}})
   => (contains-in {:a {:dependencies [['org.clojure/clojure string?]]}})"
  {:added "3.0"}
  [packages]
  (map/map-vals (fn [{:keys [dependencies] :as package}]
                  (->> dependencies
                       (map (fn [artifact]
                              (let [rep     (artifact/artifact :rep artifact)
                                    version (if (empty? (:version rep))
                                              (deps/current-version rep)
                                              (:version rep))]
                                (artifact/artifact :coord (assoc rep :version version)))))
                       (assoc package :dependencies)))
                packages))

(defn collect-linkages
  "collects all imports and exports of a package
   
   (-> -packages-
       (collect-entries -lookups-)
       (collect-linkages -lookups-)
       (get 'hara/core.version)
       (select-keys [:imports :exports]))
   => '{:imports #{[:clj hara.function]},
        :exports #{[:clj hara.core.version]}}"
  {:added "3.0"}
  [packages lookups]
  (map/map-vals (fn [{:keys [entries] :as package}]
                  (->> entries
                       (map (fn [entry]
                              (file-linkage (get-in lookups entry))))
                       (apply merge-with set/union)
                       (merge package)))
                packages))

(defn collect-internal-deps
  "collects all internal dependencies
 
   (-> -packages-
       (collect-entries -lookups-)
       (collect-linkages -lookups-)
       (collect-internal-deps)
       (get-in ['hara/test :internal]))
   => '#{hara/io.file hara/base hara/function.task hara/io.project}"
  {:added "3.0"}
  [packages]
  (let [packages (map/map-vals (fn [{:keys [name imports] :as package}]
                                 (->> (dissoc packages name)
                                      (keep (fn [[k {:keys [exports]}]]
                                              (if-not (empty? (set/intersection imports exports))
                                                k)))
                                      (set)
                                      (assoc package :internal)))
                               packages)
        _  (sort/topological-sort (map/map-vals :internal packages))]
    packages))

(defn collect-transfers
  "collects all files that are packaged
 
   (-> -packages-
       (collect-entries -lookups-)
       (collect-linkages -lookups-)
       (collect-internal-deps)
       (collect-transfers -lookups- (project/project))
       (get-in ['hara/core.version :files])
       sort)
   => (contains-in [[string? \"hara/core/version.clj\"]])"
  {:added "3.0"}
  [packages lookups project]
  (map/map-vals (fn [{:keys [entries bundle] :as package}]
                  (let [efiles (map    (fn [[suffix ns :as entry]]
                                         (let [file (get-in lookups entry)]
                                           [file (str (-> (str ns)
                                                          (.replaceAll  "\\." "/")
                                                          (.replaceAll  "-" "_"))
                                                      "."
                                                      (name suffix))]))
                                       entries)
                        bfiles (mapcat (fn [{:keys [include path]}]
                                         (mapcat (fn [b]
                                                   (let [base (fs/path (:root project) path)]
                                                     (->> (fs/select (fs/path base b)
                                                                     {:include [fs/file?]})
                                                          (map (juxt str #(str (fs/relativize base %)))))))
                                                 include))
                                       bundle)]
                    (assoc package :files (concat efiles bfiles))))
                packages))

(defn collect
  "cellects all information given lookups and project
 
   (-> (collect -packages- -lookups- (project/project))
       (get 'hara/core.version))"
  {:added "3.0"}
  ([]
   (let [project (project/project)
         lookups (create-file-lookups project)]
     (collect (read-packages) lookups project)))
  ([packages lookups project]
   (collect packages lookups project nil))
  ([packages lookups project {:keys [overlapped missing]
                              :or {overlapped :error
                                   missing :none}}]
   (let [packages (collect-entries packages lookups)
         _        (let [entries (missing-entries packages lookups)]
                    (case missing
                      :warn  (if (seq entries)
                               (println "Missing entries" entries))
                      :error (if (seq entries)
                               (throw (ex-info "Missing entries" {:entries entries})))
                      true))
         _        (let [entries (overlapped-entries packages)]
                    (case overlapped
                      :warn  (if (seq entries)
                               (println "Overlapped entries" entries))
                      :error (if (seq entries)
                               (throw (ex-info "Overlapped entries" {:entries entries})))
                      true))
         packages (-> packages
                      (collect-linkages lookups)
                      (collect-internal-deps)
                      (collect-external-deps)
                      (collect-transfers lookups project))]
     packages)))
