(ns hara.deploy.task
  (:require [hara.config :as config]
            [hara.config.gpg :deps true]
            [hara.data :as data]
            [hara.deploy.linkage :as linkage]
            [hara.io.project :as project]
            [hara.function.task :as task]
            [hara.lib.aether :as aether]))

(def +config+ "config/deploy.edn")

(defn make-project
  "makes a maven compatible project
 
   (make-project)
   => map?"
  {:added "3.0"}
  ([]
   (make-project nil))
  ([_]
   (let [project (project/project)]
     (assoc project
            :aether (aether/aether)
            :deploy (or (config/resolve (:deploy project))
                        (config/load +config+))))))

(defn select-manifest
  "selects all related manifests
 
   (select-manifest {:a {:internal #{:b}}
                     :b {:internal #{:c}}
                     :c {:internal #{}}
                     :d {:internal #{}}}
                    [:a])
   => {:a {:internal #{:b}}
       :b {:internal #{:c}}
       :c {:internal #{}}}"
  {:added "3.0"}
  [packages manifest]
  (if (= manifest :all)
    packages
    (let [find-deps (fn find-deps
                      [lookup deps entry]
                      (when-not (get @deps entry)
                        (swap! deps conj entry)
                        (doseq [e (get lookup entry)]
                          (find-deps lookup deps e))))
          deps   (atom #{})
          lookup (data/map-vals :internal packages)
          _  (doseq [entry manifest]
               (find-deps lookup deps entry))]
      (select-keys packages @deps))))

(defn make-linkages
  "creates linkages
 
   (make-linkages (make-project))
   => map?"
  {:added "3.0"}
  ([project]
   (make-linkages (:tag project)
                  project
                  (:deploy project)))
  ([tag project {:keys [packages] :as deploy}]
   (let [tag    (or tag :public)
         {:keys [type repository manifest collect]} (get-in deploy [:releases tag])
         packages (data/map-entries (fn [[k entry]]
                                      [k (assoc entry :name k)])
                                    packages)
         lookups  (linkage/create-file-lookups project)
         linkages (linkage/collect packages lookups project collect)
         linkages (select-manifest linkages manifest)]
     linkages)))

(def +main+
  {:construct {:input    (fn [_] :list)
               :lookup   (fn [task project]
                           (make-linkages project))
               :env      make-project}
   :params    {:print  {:item    true
                        :result  true
                        :summary true}
               :return :summary}
   :arglists '([] [pkg] [pkg params] [pkg params project] [pkg params lookup project])
   :main      {:count 4}})

(def +main+
  {:construct {:input    (fn [_] :list)
               :lookup   (fn [task project]
                           (make-linkages project))
               :env      make-project}
   :params    {:print  {:item    true
                        :result  true
                        :summary true}
               :return :summary}
   :arglists '([] [pkg] [pkg params] [pkg params project] [pkg params lookup project])
   :main      {:count 4}})

(defmethod task/task-defaults :deploy.linkage
  [_]
  (merge +main+
         {:item      {:list    (fn [lookup _] (vec (sort (keys lookup))))}
          :result    {:columns [{:key    :key
                                 :align  :left}
                                {:key    :time
                                 :align  :left
                                 :length 10
                                 :format "%d ms"
                                 :color  #{:bold}}]}
          :summary  {:aggregate {}}}))

(comment

  (:deploy (make-project nil))
  )
