(ns hara.deploy.maven
  (:require [hara.deploy.maven.command :as command]
            [hara.deploy.maven.package :as package]
            [hara.deploy.linkage :as linkage]
            [hara.io.project :as project]
            [hara.function :refer [definvoke]]
            [hara.function.task :as task]
            [hara.lib.aether :as aether]))

(defn create-linkages [{:keys [tag root deploy] :as project}]
  (let [tag    (or tag :release)
        {:keys [file config type collect]} (get deploy tag)
        packages (linkage/read-packages {:file file :root root})
        lookups  (linkage/create-file-lookups project)
        linkages (linkage/collect packages lookups project collect)]
    linkages))

(def +main+
  {:construct {:input    (fn [_] :list)
               :lookup   (fn [task project]
                           (create-linkages project))
               :env      (fn [_]
                           (assoc (project/project) :aether (aether/aether)))}
   :params    {:print  {:item    true
                        :result  true
                        :summary true}
               :return :summary}
   :arglists '([] [pkg] [pkg params] [pkg params project] [pkg params lookup project])
   :main      {:count 4}})

(defmethod task/task-defaults :deploy
  [_]
  (merge +main+
         {:item      {:list     (fn [lookup _] (vec (sort (keys lookup))))
                      :display  (fn [data] (vec (sort (map :extension data))))}
          :result    {:keys    {:artifacts (fn [data]
                                             (let [{:keys [version group artifact]} (first data)
                                                   extensions (sort (map (comp str :extension) data))]
                                               [group version extensions]))}
                      :columns [{:key    :key
                                 :align  :left}
                                {:key    :artifacts
                                 :align  :left
                                 :length 60
                                 :color  #{:bold}}
                                {:key    :time
                                 :align  :left
                                 :length 10
                                 :format "%d ms"
                                 :color  #{:bold}}]}
          :summary  {:aggregate {}}}))

(defmethod task/task-defaults :deploy.linkage
  [_]
  (merge +main+
         {:item      {:list    (fn [lookup _] (vec (sort (keys lookup))))}
          :result    {:columns [{:key    :key
                                 :align  :left}
                                {:key    :time
                                 :align  :left
                                 :length 10
                                 :format "%d ms"
                                 :color  #{:bold}}]}
          :summary  {:aggregate {}}}))

(defmethod task/task-defaults :deploy.package
  [_]
  (merge +main+
         {:item      {:list     (fn [lookup _] (vec (sort (keys lookup))))
                      :display  (juxt (comp count :files) :pom)}
          :result    {:keys    {:jar :jar
                                :pom :pom
                                :packaged (comp count :files)}
                      :columns [{:key    :key
                                 :align  :left}
                                {:key    :packaged
                                 :align  :left
                                 :format "(%d)"
                                 :length 10
                                 :color  #{:bold}}
                                {:key    :jar
                                 :align  :left
                                 :length 40
                                 :color  #{:bold}}
                                {:key    :time
                                 :align  :left
                                 :length 10
                                 :format "%d ms"
                                 :color  #{:bold}}]}
          :summary  {:aggregate {:packaged   [:packaged + 0]}}}))

(definvoke linkage
  "creates the linkages required for deployment"
  {:added "3.0"}
  [:task {:template :deploy.linkage
          :params {:title "CREATES ALL LINKAGE FILES"}
          :main {:fn package/linkage}}])

(definvoke package
  "packages files in the interim directory
 
   (package '[hara])"
  {:added "3.0"}
  [:task {:template :deploy.package
          :params {:title "PACKAGE INTERIM FILES"}
          :main {:fn package/package}}])

(definvoke clean
  "cleans the interim directory of packages
 
   (clean :all)"
  {:added "3.0"}
  [:task {:template :deploy.linkage
          :params {:title "CLEAN ALL INTERIM FILES"}
          :main {:fn command/clean}}])

(definvoke install
  "installs packages to the local `.m2` repository
 
   (install '[hara])"
  {:added "3.0"}
  [:task {:template :deploy
            :params {:title "INSTALL PACKAGES"}
          :main {:fn command/install}}])

(definvoke install-secure
  "installs signed packages to the local `.m2` repository"
  {:added "3.0"}
  [:task {:template :deploy
          :params {:title "INSTALL SIGNED PACKAGES"}
          :main {:fn command/install-secure}}])

(definvoke deploy
  "deploys packages to a maven repository
 
   (deploy '[spirit]
           {:repository {:id \"hara\"
                        :url \"https://maven.hara.io\"
                         :authentication {:username \"hara\"
                                          :password \"hara\"}}})"
  {:added "3.0"}
  [:task {:template :deploy
          :params {:title "DEPLOY PACKAGES"}
          :main {:fn command/deploy}}])

(comment
  (./code:scaffold)
  (install '[hara])
  (package '[hara])
  (linkage '[hara])
  (clean '[hara])
  
  (get-in (create-env (project/project))
          '[:linkages hara/publish :internal])
  #{hara/code hara/io.file hara/lib.jsoup hara/base hara/string.text hara/function.task hara/io.project hara/test}
  

  (get-in (create-env (assoc (project/project) :tag :public))
          '[:linkages hara/publish :internal])
  #{hara/code hara/io.file hara/lib.jsoup hara/base hara/string.text hara/function.task hara/io.project hara/test})
