(ns hara.data.schema.impl
  (:require [hara.data.base.tree :as tree]
            [hara.data.base.map :as map]
            [hara.data.schema.alias :as alias]
            [hara.data.schema.base :as base]
            [hara.data.schema.find :as find]
            [hara.data.schema.ref :as ref]))

(defn simplify
  "helper function for easier display of spirit schema
   (simplify {:account/name  [{:type :long}]
              :account/email [{:type :string
                               :cardinality :many}]
              :email/accounts [{:type :ref
                                :cardinality :many
                                :ref {:ns :account}}]})
   => {:email {:accounts :&account<*>}
       :account {:email :string<*> :name :long}}"
  {:added "3.0"}
  [flat]
  (->> flat
       (reduce-kv (fn [out k [attr]]
                    (let [card-str (condp = (:cardinality attr)
                                     :many "<*>"
                                     "")
                          type-str (condp = (:type attr)
                                     :ref (str "&" (name (-> attr :ref :ns)))
                                     (name (:type attr)))]
                      (assoc out k (keyword (str type-str card-str)))))
                  {})
       (tree/treeify-keys)))

(defrecord Schema [flat tree lu]
  Object
  (toString [this]
    (str "#schema" {:tables (count tree)})))

(defmethod print-method Schema
  [v ^java.io.Writer w]
  (.write w (str v)))

(defn create-lookup
  "lookup from flat schema mainly for reverse refs
   (create-lookup
    {:account/name   [{}]
     :account/email  [{}]
     :email/accounts [{:ident :email/accounts
                       :type :ref
                       :ref {:type :reverse
                             :key :account/_email}}]})
   => {:email/accounts :email/accounts
      :account/_email :email/accounts
       :account/email :account/email
       :account/name :account/name}"
  {:added "3.0"}
  [fschm]
  (reduce-kv (fn [out k [attr]]
               (cond (find/is-reverse-ref? attr)
                     (assoc out (-> attr :ref :key) k k k)

                     :else
                     (assoc out k k)))
             {} fschm))

(defn create-flat-schema
  "creates a flat schema from an input map
   (create-flat-schema {:account {:email [{:type    :ref
                                           :ref     {:ns  :email}}]}})
   => {:email/accounts [{:ident :email/accounts
                         :type :ref
                         :cardinality :many
                         :ref {:ns :account
                               :type :reverse
                               :key :account/_email
                              :val :accounts
                               :rval :email
                               :rkey :account/email
                               :rident :account/email}}]
       :account/email [{:ident :account/email
                        :type :ref
                        :cardinality :one
                        :ref  {:ns :email
                               :type :forward
                               :key :account/email
                               :val :email
                               :rval :accounts
                               :rkey :account/_email
                               :rident :email/accounts}}]}"
  {:added "3.0"}
  ([m]
   (create-flat-schema m (base/all-auto-defaults base/base-meta)))
  ([m defaults]
   (let [fschm (->> (tree/flatten-keys-nested m)
                    (map base/attr-add-ident)
                    (map #(base/attr-add-defaults % defaults)) ;; meta/all-auto-defaults
                    (into {}))]
     (merge fschm
            (ref/ref-attrs fschm)
            (alias/alias-attrs fschm)))))

(defn vec->map
  "turns a vec schema to a map
 
   (vec->map [:account [:id    {:type :long}
                        :name  {:type :text}]])
   => {:account {:id [{:type :long, :order 0}],
                 :name [{:type :text, :order 1}]}}"
  {:added "3.0"}
  ([v]
   (->> (apply hash-map v)
        (map/map-vals (fn [columns]
                        (->> columns
                             (partition 2)
                             (map-indexed (fn [i [name attrs]]
                                            [name [(assoc attrs :order i)]]))
                             (into {})))))))

(defn schema-map
  "creates a schema from a map
 
   (-> (schema-map {:account/name   [{}]
                    :account/email  [{:ident   :account/email
                                     :type    :ref
                                      :ref     {:ns  :email}}]})
       :flat
       simplify)"
  {:added "3.0"}
  ([m]
   (schema-map m (base/all-auto-defaults base/base-meta)))
  ([m defaults]
   (let [flat (create-flat-schema m defaults)
         tree (tree/treeify-keys flat)
         lu   (create-lookup flat)]
     (Schema. flat tree lu))))

(defn schema
  "creates an extended schema for use by spirit
 
   (-> (schema [:account [:name  {}
                          :email {:type :ref :ref {:ns :email}}]])
       :flat
       simplify)
   => {:email {:accounts :&account<*>}
       :account {:email :&email
                 :name :string}}"
  {:added "3.0"}
  ([x]
   (schema x (base/all-auto-defaults base/base-meta)))
  ([x defaults]
   (cond (vector? x)
         (let [schema (-> (vec->map x)
                          (schema-map defaults))]
           (assoc schema :vec x))

         (map? x)
         (schema-map x defaults))))

(defn schema?
  "checks if object is a schema
 
   (schema? (schema {:user/email [{}]}))
   => true"
  {:added "3.0"}
  [obj]
  (instance? Schema obj))
