(ns hara.data.pipeline.require
  (:require [hara.core.base.check :as check]
            [hara.event :as event]
            [hara.function :as fn]))

(defn process-require
  "Checks for correct entry
 
   (require/process-require {:name :checked}
                            :no-required
                            {:name \"Chris\"}
                            [:account]
                            (-> (schema/schema examples/account-name-age-sex)
                                :tree
                                :account)
                            {})
  => {:name \"Chris\"}"
  {:added "3.0"}
  [req require-key tdata nsv tsch datasource]
  (if-let [[k v] (first req)]
    (cond (= v :checked)
          (do (if (not (get tdata k))
                (event/raise [:normalise require-key {:nsv (conj nsv k) :data tdata}]
                             (str "PROCESS_REQUIRE: key " (conj nsv k) " is not present")))
              (recur (next req) require-key tdata nsv tsch datasource))

          (fn? v)
          (let [subdata (get tdata k)
                flag (fn/op v subdata datasource)]
            (do (if (and (or (= flag :checked)
                             (true? flag))
                         (nil? subdata))
                  (event/raise [:normalise require-key {:nsv (conj nsv k) :data tdata}]
                               (str "PROCESS_REQUIRE: key " (conj nsv k) " is not present")))
                (recur (next req) require-key tdata nsv tsch datasource)))

          (and (-> tsch (get k) vector?)
               (-> tsch (get k) first :type (= :ref)))
          (recur (next req) require-key tdata nsv tsch datasource)

          :else
          (let [subdata (get tdata k)]
            (cond (nil? subdata)
                  (event/raise [:normalise require-key {:nsv (conj nsv k) :data tdata}]
                               (str "PROCESS_REQUIRE: key " (conj nsv k) " is not present"))

                  (check/hash-map? subdata)
                  (process-require v require-key (get tdata k) (conj nsv k) (get tsch k) datasource))
            (recur (next req) require-key tdata nsv tsch datasource)))
    tdata))

(defn wrap-model-pre-require
  "Checks for data across elements and schema pre transforms
   (pipeline/normalise
    {:account/orders #{{:number 1}
                       {:number 2}}}
    {:schema (schema/schema examples/account-orders-items-image)
     :pipeline {:pre-require
                {:account {:orders
                           {:number :checked}}}}}
    *wrappers*)
   => {:account {:orders #{{:number 1}
                           {:number 2}}}}
   
   (pipeline/normalise
    {:account/orders #{{:items {:name \"stuff\"}}
                       {:number 2}}}
    {:schema (schema/schema
              examples/account-orders-items-image)
     :pipeline {:pre-require
                {:account
                 {:orders {:number :checked}}}}}
    *wrappers*)
   => (throws-info {:data {:items {:name \"stuff\"}}
                   :nsv [:order :number]
                    :no-required true})"
  {:added "3.0"}
  [f]
  (fn [tdata tsch nsv interim fns datasource]
    (let [req (:pre-require interim)]
      (process-require req :no-required tdata nsv tsch datasource)
      (f tdata tsch nsv interim fns datasource))))

(defn wrap-model-post-require
  "Checks for data across elements and schema post transforms
 
   (pipeline/normalise
    {:account/name \"Chris\"}
    {:schema (schema/schema examples/account-name-age-sex)
     :pipeline {:post-require {:account {:name :checked}}}}
    {:normalise [require/wrap-model-post-require]})
   => {:account {:name \"Chris\"}}
 
   (pipeline/normalise
    {:account/age 10}
    {:schema (schema/schema examples/account-name-age-sex)
     :pipeline {:post-require
                {:account {:name :checked}}}}
    {:normalise [require/wrap-model-post-require]})
   => (throws-info {:nsv [:account :name]
                    :no-required true})"
  {:added "3.0"}
  [f]
  (fn [tdata tsch nsv interim fns datasource]
    (let [req (:post-require interim)
          output (f tdata tsch nsv interim fns datasource)]
      (process-require req :no-required output nsv tsch datasource))))
