(ns hara.data.pipeline.allow
  (:require [hara.core.base.check :as check]
            [hara.event :as event]
            [hara.string :as string]))

(defn wrap-branch-model-allow
  "Works together with wrap-attr-model-allow to control access to data
   (pipeline/normalise {:account/name \"Chris\"}
                        {:schema (schema/schema examples/account-name-age-sex)
                         :pipeline {:allow {}}}
                        *wrappers*)
   => (throws-info {:data {:name \"Chris\"}
                    :key-path [:account]
                    :normalise true
                    :not-allowed true
                    :nsv [:account]})
   
   (pipeline/normalise {:account/name \"Chris\"}
                        {:schema (schema/schema examples/account-name-age-sex)
                         :pipeline {:allow {:account {:name :checked}}}}
                        *wrappers*)
   => {:account {:name \"Chris\"}}
   "
  {:added "3.0"}
  [f]
  (fn [subdata subsch nsv interim fns datasource]
    (let [suballow (:allow interim)
          nsubdata  (if (nil? suballow)
                      (event/raise [:normalise :not-allowed
                                    {:data subdata :nsv nsv :key-path (:key-path interim)}]
                                   (str "WRAP_BRANCH_MODEL_ALLOW: key " nsv " is not accessible."))
                      subdata)]
      (f nsubdata subsch nsv interim fns datasource))))

(defn wrap-attr-model-allow
  "wrapper function for only allowing values defined to be included
   
   (pipeline/normalise {:account {:name \"Chris\"
                                  :age 10}}
                       {:schema (schema/schema examples/account-name-age-sex)
                        :pipeline {:allow {:account {:name :checked}}}}
                       *wrappers*)
   => throws"
  {:added "3.0"}
  [f]
  (fn [subdata [attr] nsv interim fns datasource]
    (let [suballow (:allow interim)]
      (cond (= (:type attr) :ref)
            (cond (= suballow :yield)
                  (let [ynsv   (string/path-split (-> attr :ref :ns))
                        tmodel (get-in datasource (concat [:pipeline :allow] ynsv))]
                    (f subdata [attr] ynsv (assoc interim :allow tmodel) fns datasource))

                  (or (= suballow :id) (check/hash-map? suballow))
                  (f subdata [attr] nsv interim fns datasource)

                  :else
                  (event/raise [:normalise :not-allowed
                                {:data subdata :nsv nsv :key-path (:key-path interim)}]
                               (str "WRAP_ATTR_MODEL_ALLOW: " nsv " is not accessible")))

            (or (nil? suballow) (not= suballow :checked))
            (let [nsubdata
                  (event/raise [:normalise :not-allowed
                                {:data subdata :nsv nsv :key-path (:key-path interim)}]
                               (str "WRAP_ATTR_MODEL_ALLOW: " nsv " is not accessible"))]
              (f nsubdata [attr] nsv interim fns datasource))

            :else
            (f subdata [attr] nsv interim fns datasource)))))
