(ns hara.code.framework.test.fact
  (:require [hara.code.framework.common :as common]
            [hara.code.query :as query]
            [hara.code.block :as block]
            [hara.code.navigate :as nav]))

(defn gather-fact-body
  "helper function for `gather-fact`
   (-> \"(\\n  (+ 1 1) => 2\\n  (long? 3) => true)\"
       nav/parse-string
       nav/down
       (gather-fact-body)
       (docstring/->docstring))
   => \"\\n  (+ 1 1) => 2\\n  (long? 3) => true\""
  {:added "3.0"}
  ([nav]
   (gather-fact-body nav []))
  ([nav output]
   (cond (nil? (nav/block nav)) output

         (and (= :meta (nav/tag nav))
              (-> nav nav/down nav/position-right nav/value (= :hidden)))
         output

         (query/match nav string?)
         (recur (nav/right* nav)
                (conj output (common/gather-string nav)))

         :else
         (recur (nav/right* nav) (conj output (nav/block nav))))))

(defn gather-fact
  "Make docstring notation out of fact form
   (-> \"^{:refer example/hello-world :added \\\"0.1\\\"}
        (fact \\\"Sample test program\\\"\\n  (+ 1 1) => 2\\n  (long? 3) => true)\"
       (nav/parse-string)
       nav/down nav/right nav/down nav/right
       (gather-fact)
       (update-in [:test] docstring/->docstring))
   => (just-in {:form  'fact
                :ns    'example,
               :var   'hello-world,
                :refer 'example/hello-world
                :added \"0.1\",
                :line  {:row 2, :col 8, :end-row 4, :end-col 21}
                :intro \"Sample test program\",
                :test  \"\\n  (+ 1 1) => 2\\n  (long? 3) => true\"})"
  {:added "3.0"}
  [nav]
  (if-let [mta (common/gather-meta nav)]
    (let [exp (nav/value nav)
          [intro nnav] (if (string? exp)
                          [exp (if (nav/right nav)
                                 (nav/right* nav))]
                          ["" nav])]
      (assoc mta
             :form  (-> nav nav/left nav/value)
             :line  (nav/line-info (nav/up nav))
             :test  (if nnav
                      (gather-fact-body nnav)
                      [])
             :intro intro))))

(defmethod common/test-frameworks 'midje.sweet  [_]  :fact)
(defmethod common/test-frameworks 'hara.test    [_]  :fact)

(defmethod common/analyse-test :fact
  ([type nav]
   (let [fns  (query/$ nav [(#{fact comment} | & _)] {:return :zipper :walk :top})]
     (->> (keep gather-fact fns)
          (reduce (fn [m {:keys [ns var class test intro line form] :as meta}]
                    (-> m
                        (update-in [ns var]
                                   assoc
                                   :ns ns
                                   :var var
                                   :class class
                                   :test  {:path common/*path*
                                           :form form
                                           :code test
                                           :line line}
                                   :meta  (apply dissoc meta common/+test-vars+)
                                   :intro intro)))
                  {})))))
