(ns gv.core
  (:require [clojure.java.io :as io]
            [clojure.java.shell :as sh]
            [malli
             [core :as m]
             [util :as mu]]
            [medley.core :refer :all]
            [hiccup.core :refer [html]]
            [seesaw [core :refer :all]]
            [gv.view :refer [view-dot! save-dot!]])
  (:import [javax.swing ImageIcon]))

;; TODO: improve error messages when a data does not conform to
;; an appropriate schema.

;; utility functions and macros.

(def ^{:private? true} nl "\n")

(def ^{:private? true} linesep (str ";" nl))

(def ^{:private? true} comma ",")

(defn- wrap [s] (format "\"%s\"" s))

(defn- branket [s] (format "[%s]" s))

(defn- ->str [x] (if (keyword? x) (name x) (str x)))

(defn- attr-str
  "Returns coll of vector [attr-key attr-value]."
  [map-or-seq]
  (cond
    (map? map-or-seq)
    (map (fn [[k v]] (str (->str k) " = " (-> v ->str wrap))) map-or-seq)
    
    (sequential? map-or-seq)
    (map (fn [[k v]] (str (->str k) " = " (->str v)))
         (partition 2 map-or-seq))))

(defmacro check
  {:private? true
   :arglists '([schema value & more])
   :doc "Throws an ExceptionInfo if `value` does not conform to `schema`."}
  [& forms]
  (when forms
    (let [[s x & more] forms]
      `(do
         (when-not (m/validate ~s ~x)
           (throw (ex-info "The input does not conform to the schema."
                           {:input ~x
                            :schema ~s})))
         (check ~@more)))))

;; define data

(def ^{:private true} label-schema
  (m/schema [:or string? keyword? number?]))

(def ^{:private true} attr-schema
  (m/schema [:map-of keyword? label-schema]))

(def ^{:private true} node-schema
  (m/schema [:orn
             [:unit label-schema]
             [:coll [:cat label-schema attr-schema]]]))

(def ^{:private true} edge-schema
  (m/schema [:catn
             [:n [:repeat {:min 2} node-schema]]
             [:a [:? attr-schema]]]))

(def ^{:private true} graph-schema
  (m/schema
   [:schema
    {:registry
     {::graph [:map
               [:nodes {:optional true} [:sequential node-schema]]
               [:edges {:optional true} [:sequential edge-schema]]
               [:type {:optional true} [:enum :directed :undirected]]
               [:clusters {:optional true} [:sequential [:ref ::graph]]]]}}
    ::graph]))

(defn- node->str [x]
  (case (key (m/parse node-schema x))
    :unit (str x)
    :coll (let [[label m] x
                a (reduce str (interpose ", " (attr-str m)))]
            (format "%s [%s]" label a))))

(defn- edge->str [x sep]
  (let [{:keys [n a]} (m/parse edge-schema x)
        nodes (m/unparse [:* node-schema] n)
        nodes (->> nodes (map node->str) (interpose sep) (reduce str))
        a (some->> a attr-str (interpose ", ") (reduce str) branket)]
    (str nodes  " " a)))

(defn- node-attr->str [m]
  (if (not-empty m)
    (format "node [%s]" (apply str (interpose ", " (attr-str m))))
    ""))

(defn- edge-attr->str [m]
  (if (not-empty m)
    (format "edge [%s]" (apply str (interpose ", " (attr-str m))))
    ""))

(defn- graph-attr->str [g]
  (let [{:keys [node-attr edge-attr]} g
        m (dissoc g :node-attr :edge-attr :nodes :edges :clusters
                  :type)
        coll (conj (attr-str m)
                   (edge-attr->str edge-attr) (node-attr->str node-attr))
        coll (filter not-empty coll)]
    (apply str (map #(str % linesep) coll))))

(defn- subgraph-label [] (str "subgraph cluster_" (gensym)))

(defn- join-components [sep coll]
  (let [coll (filter not-empty coll)]
    (apply str (map str coll (repeat sep)))))

(defn- graph->str [label sep g]
  (let [{:keys [nodes edges clusters]} g
        nodes (join-components linesep (map node->str nodes))
        edges (join-components linesep (map #(edge->str % sep) edges))
        clusters (join-components linesep
                                  (map
                                   #(graph->str (subgraph-label) sep % )
                                   clusters))
        attr (graph-attr->str g)
        all (join-components nl [attr nodes edges clusters])]
    (format "%s{%s}" label (str nl all))))

(defn dot [g]
  (check graph-schema g)
  (let [{:keys [type]} g
        directed? (= type :directed)
        sep (if directed? " -> " " -- ")
        label (if directed? "digraph" "graph")]
    (graph->str label sep g)))

(defn- new-graph [type args]
  (let [ret (assoc (apply hash-map args) :type type)]
    (check graph-schema ret)
    ret))

(defn digraph
  "Creates a directed graph.
  Example:

  (digraph
  :nodes [\"start\" [\"end\" {:shape \"box\"}]]
  :edges [[\"start\" 1] [\"start\" 5] [4 \"end\"] [8 \"end\"]]
  :clusters [(digraph :edges [[1 2 3 4]]
                     :bgcolor \"lightgrey\"
                     :label \"process 1\")
            (digraph :edges [[5 6 7 8]]
                     :label \"process 2\")])"
  [& args] (new-graph :directed args))

(defn graph
  "Creates an undirected graph.
  Example:
  
  (graph
  :nodes [\"start\" [\"end\" {:shape \"box\"}]]
  :edges [[\"start\" 1] [\"start\" 5] [4 \"end\"] [8 \"end\"]]
  :clusters [(digraph :edges [[1 2 3 4]]
                     :bgcolor \"lightgrey\"
                     :label \"process 1\")
            (digraph :edges [[5 6 7 8]]
                     :label \"process 2\")])"
  [& args] (new-graph :undirected args))

(defn view!
  "Visualizes a graph."
  [g &{:keys [scale] :or {scale 2}}]
  (view-dot! (dot g) scale))

(defn save-graph!
  "Saves an image in a file.

  Example:
  (save-graph! (graph :nodes [1 2 3]) \"graph.png\")"
  [g filename]
  (save-dot! (dot g) filename))

(comment

  (let [g (graph
           :nodes ["start" ["end" {:shape "box"}]]
           :edges [["start" 1] ["start" 5] [4 "end"] [8 "end"]]
           :clusters [(digraph :edges [[1 2 3 4]]
                               :bgcolor "lightgrey"
                               :label "process 1")
                      (digraph :edges [[5 6 7 8]]
                               :label "process 2")])]
    (view! g))

  )



