(ns genesis.util
  (:require [clojure.data.codec.base64 :as b64]
            [clojure.spec.alpha :as s]
            [orchestra.spec.test :refer [instrument]])
  (:import clojure.lang.Var))

(defn base64-encode [s]
  (String. (b64/encode (.getBytes s "UTF-8"))))

(defn base64-decode [^String s]
  (String. (b64/decode (.getBytes s "UTF-8"))))

(defn validate!
  ([s x]
   (validate! s x "schema does not conform"))
  ([s x message]
   (when-not (s/valid? s x)
     (throw (ex-info (str "validation failed:" (s/form s) (s/explain-str s x))
                     {:spec s
                      :data x
                      :explain (s/explain-data s x)})))
   (or x true)))

(defn duplicates
  "The opposite of distinct, returns items that appear multiple times in coll"
  [coll]
  (loop [seen #{}
         dupes #{}
         coll coll]
    (if (seq coll)
      (let [c (first coll)]
        (if (contains? seen c)
          (recur (conj seen c) (conj dupes c) (rest coll))
          (recur (conj seen c) dupes (rest coll))))
      dupes)))

(defn deref? [x]
  (instance? clojure.lang.IDeref x))

(defn ref? [x]
  (instance? clojure.lang.IRef x))

(defn ref-of
  "takes a spec. Returns a predicate"
  [s]
  (fn [x]
    (and (ref? x) (s/valid? s @x))))

(defn unwrap!
  "If x is a map with one key, return the value, else assert false"
  [x]
  (if (and (map? x) (= 1 (count x)))
    (-> x first val)
    (assert false (format "can't unwrap: %s" x))))

(defn maybe-unwrap [x]
  (if (and (map? x) (= 1 (count x)))
    (unwrap! x)
    x))

(defn get-by-list-property
  "Return a resource get fn, by calling list and returning first instance w/ matching property"
  [{:keys [list-fn property-key]}]
  (fn [context property-value]
    (->>
     (list-fn context)
     (filter (fn [x]
               (assert (contains? (:properties x) property-key))
               (= property-value (-> x :properties (get property-key)))))
     first)))

(defn instrument-ns
  ([]
   (instrument-ns *ns*))
  ([ns]
   (println "instrumenting" ns)
   (s/check-asserts true)
   (->> ns
        (ns-publics)
        (vals)
        (mapv (fn [^Var v]
                (symbol (str (.ns v) "/" (.sym v)))))
        (instrument))
   nil))
