(ns genesis.providers.aws.route53.record-set
  (:require [amazonica.aws.route53 :as r53]
            [circle.wait-for :refer (wait-for)]
            [clj-time.core :as time]
            [genesis.providers.aws.ec2 :as gec2]
            [genesis.providers.aws.route53.hosted-zone :as hosted-zone]
            [clojure.set :as set]
            [clojure.spec.alpha :as s]
            [clojure.data :as data]
            [genesis.core :as g :refer [defresource]]
            [genesis.util :refer [base64-encode base64-decode validate!]]))

(defn proper-dns-name? [name]
  (re-find #"\.$" name))

(s/def ::name (s/and string? proper-dns-name?))
(s/def ::type string?)
(s/def ::value string?)

(s/def ::dnsname string?)
(s/def ::evaluate-target-health boolean?)
(s/def ::alias-target (s/keys :req-un [::dnsname
                                       ::hosted-zone-id
                                       ::evaluate-target-health]))

(s/def ::resource-record (s/keys :req-un [::value]))
(s/def ::resource-records (s/coll-of ::resource-record))

(s/def ::ttl nat-int?)

(s/def ::record-set (s/keys :req-un [::name
                                     ::type]
                            :opt-un [::resource-records
                                     ::alias-target
                                     ::ttl]))


(defn list-record-sets* [creds hz-id & [{:keys [start-record-identifier]}]]
  {:pre [(string? hz-id)]}
  (let [resp (r53/list-resource-record-sets creds (merge {:hosted-zone-id hz-id}
                                                         (when start-record-identifier
                                                           {:start-record-identifier start-record-identifier})))
        rrs (:resource-record-sets resp)]
    (if (:is-truncated resp)
      (do
        (assert (:next-record-identifier resp))
        (lazy-cat rrs (list-record-sets* creds hz-id {:start-record-identifier (:next-record-identifier resp)})))
      rrs)))

(defn make-identity [zone type name]
  (str zone " " type " " name))

(defn list-record-sets [context]
  (let [creds (-> context :aws :creds)]
    (->> (hosted-zone/list-hosted-zones context)
         (mapcat (fn [hz]
                   (let [hosted-zone-id (-> hz :properties :id)
                         rrs (list-record-sets* creds hosted-zone-id)]
                     (map (fn [rrs]
                            {:identity (make-identity hosted-zone-id (:type rrs) (:name rrs))
                             :properties (assoc rrs :hosted-zone-id hosted-zone-id)}) rrs)))))))

(def get-record (gec2/get-by-identity list-record-sets))

(defn wait-for-change [context change]
  (let [creds (-> context :aws :creds)
        change-id (-> change :change-info :id)]
    (println "route53: waiting for change" (:change-info change))
    (assert change-id)
    (wait-for {:timeout (time/minutes 10)
               :sleep (time/seconds 5)}
              (fn []
                (-> (r53/get-change creds {:id change-id})
                    :change-info
                    :status
                    (= "INSYNC"))))))

(defn create-record [context properties]
  {:post [%]}
  (validate! ::record-set properties)
  (let [hosted-zone-id (:hosted-zone-id properties)
        properties (dissoc properties :hosted-zone-id)
        creds (-> context :aws :creds)]
    (assert hosted-zone-id)
    (let [change (r53/change-resource-record-sets creds {:hosted-zone-id hosted-zone-id
                                                         :change-batch {:changes [{:action "CREATE"
                                                                                   :resource-record-set properties}]}})
          identity (make-identity hosted-zone-id (:type properties) (:name properties))]
      ;;(println "create-record identity:" identity)
      ;;(wait-for-change context change)
      (wait-for {:timeout (time/minutes 5)
                 :sleep (time/seconds 1)} #(get-record context identity)))))

(defn delete-record [context identity]
  (let [record (get-record context identity)
        creds (-> context :aws :creds)
        hosted-zone-id (-> record :properties :hosted-zone-id)
        record (if (-> record :properties :resource-records seq nil?)
                 (update-in record [:properties] dissoc :resource-records)
                 record)
        req {:hosted-zone-id hosted-zone-id
             :change-batch {:changes [{:action "DELETE"
                                       :resource-record-set (:properties record)}]}}]
    (assert record)
    (assert hosted-zone-id)
    (println "delete:" req)
    (r53/change-resource-record-sets creds req)))

(defresource :route53/record {:create create-record
                              :list list-record-sets
                              :get get-record
                              :delete delete-record})
