(ns genesis.providers.aws.rds
  (:require [clojure.spec.alpha :as s]
            [cognitect.aws.client.api :as aws]
            [genesis.providers.aws :as gaws]
            [genesis.core :as g :refer [defresource]]
            [genesis.specs :as gs]
            [genesis.util :refer [base64-encode base64-decode validate! maybe-unwrap instrument-ns unwrap!]]))

(s/def ::dbinstance-identifier string?)
(s/def ::aws-db-instance (s/keys :req-un [::DBInstanceIdentifier]))

(s/fdef ->db-instance :args (s/cat :i ::aws-db-instance) :ret ::gs/existing-instance)
(defn ->db-instance [i]
  {:resource ::database-instance
   :identity (:DBInstanceIdentifier i)
   :properties i})

(defn list-database-instances [context]
  (-> (gaws/client context :rds)
      (aws/invoke {:op :DescribeDBInstances})
      :DBInstances
      (->> (map ->db-instance))))

(s/def ::AllocatedStorage pos-int?)
(s/def ::DBInstanceClass string?)
(s/def ::DBInstanceIdentifier string?)
(s/def ::engine #{"aurora" "aurora-mysql" "aurora-postgresql" "mariadb" "mysql" "oracle-ee" "oracle-se2" "oracle-se1" "oracle-se" "postgres" "sqlserver-ee" "sqlserver-se" "sqlserver-ex" "sqlserver-web"})

(s/def ::master-user (s/and string? (partial re-find #"^\p{Alpha}.{1,62}$")))
(s/def ::master-user-password (s/and string? (partial re-find #"^.{8,128}$")))

(s/def :create-db-instance/properties (s/keys :req-un [::AllocatedStorage
                                                       ::DBInstanceClass
                                                       ::DBInstanceIdentifier
                                                       ::Engine
                                                       ::MasterUsername
                                                       ::MasterUserPassword]))

(s/fdef create-database-instance :args (s/cat :c ::gs/context :p :create-db-instance/properties) :ret ::gs/existing-instance)
(defn create-database-instance [context properties]
  (-> (gaws/client context :rds)
      (gaws/invoke! {:op :CreateDBInstance :request properties})
      :DBInstance
      ->db-instance))

(s/fdef get-database-instance :args (s/cat :c ::gs/context :i ::gs/identity) :ret (s/nilable ::gs/existing-instance))
(defn get-database-instance [context identity]
  (some->
   (gaws/client context :rds)
   (gaws/invoke! {:op :DescribeDBInstances :request {:DBInstanceIdentifier identity}})
   :DBInstances
   first
   ->db-instance))

(s/fdef delete-database-instance :args (s/cat :c ::gs/context :i ::gs/identity))
(defn delete-database-instance [context identity]
  (->
   (gaws/client context :rds)
   (gaws/invoke! {:op :DeleteDBInstance :request {:DBInstanceIdentifier identity
                                                  :SkipFinalSnapshot true}})))

(defn update-database-instance [context identity properties]
  (->
   (gaws/client context :rds)
   (gaws/invoke! {:op :ModifyDBInstance :request (assoc properties :DBInstanceIdentifier identity)})))

(instrument-ns)

(defresource ::database-instance {:list list-database-instances
                                  :create create-database-instance
                                  :get get-database-instance
                                  :delete delete-database-instance
                                  :update update-database-instance})
