(ns genesis.providers.aws.autoscaling.launch-config
  (:require [amazonica.aws.autoscaling :as autoscaling]
            [clojure.core.strint :refer (<<)]
            [clojure.spec.alpha :as s]
            [genesis.core :as g :refer [defresource]]
            [genesis.util :refer [validate! base64-encode base64-decode]]
            [genesis.providers.aws.ec2 :as gec2]))

(defn list-launch-configs [context]
  (->> (autoscaling/describe-launch-configurations (-> context :aws :creds))
       :launch-configurations
       (map (fn [lc]
              {:resource :autoscaling/launch-config
               :identity (:launch-configuration-arn lc)
               :properties (update-in lc [:user-data] base64-decode)}))))

(defn get-launch-config [context identity]
  (->> (list-launch-configs context)
       (filter (fn [lc]
                 (= identity (:identity lc))))
       first))

(s/def ::launch-configuration-name string?)
(s/def ::image-id string?)
(s/def ::instance-type string?)
(s/def ::instance-id string?)
(s/def ::iam-instance-profile string?)

(s/def ::launch-configuration (s/keys :req-un [::launch-configuration-name
                                               ::iam-instance-profile
                                               (or (and ::image-id ::instance-type)
                                                   ::instance-id)]))

(defn create-launch-config [context properties]
  (let [creds (-> context :aws :creds)
        lc (-> properties
               (gec2/with-base64-encode [:user-data]))]
    (validate! ::launch-configuration lc)
    (autoscaling/create-launch-configuration creds lc)
    (let [lc (->> (list-launch-configs context)
                  (filter (fn [lc]
                            (= (:launch-configuration-name properties) (-> lc :properties :launch-configuration-name))))
                  first)]
      (assert lc)
      (println "create launch config:" lc)
      lc)))

(defn delete-launch-config [context identity]
  (let [lc (get-launch-config context identity)
        lc-name (-> lc :properties :launch-configuration-name)
        _ (assert lc-name)
        creds (-> context :aws :creds)]
    (autoscaling/delete-launch-configuration creds {:launch-configuration-name lc-name})))

(defresource :autoscaling/launch-config {:list list-launch-configs
                                         :get get-launch-config
                                         :create create-launch-config
                                         :delete delete-launch-config})
