(ns genesis.core
  (:refer-clojure :exclude [get get-in key update])
  (:require [clojure.core :as c]
            [clojure.core.strint :refer (<<)]
            [clojure.data :as data]
            [clojure.spec.alpha :as s]
            [clojure.set :as set]
            [cheshire.core :as json]
            [genesis.resource :as resource]
            [genesis.specs :as gs]
            [genesis.state :as state]
            [genesis.topo-sort :as topo-sort]
            [genesis.util :refer [duplicates validate! deref? instrument-ns]]))


(s/fdef defresource :args (s/cat :r ::gs/resource :args (s/keys)))
(defn defresource [key args]
  (resource/create-resource key args))

(defn validate-desired [desired]
  (assert (every? :identity)))

(defn ->json [x]
  (json/generate-string x))

(defn <-json [x]
  (json/parse-string x keyword))

(def ^:dynamic *planning* false)
(def ^:dynamic *applying* false)
(def ^:dynamic *deps* nil)
(def ^:dynamic *context* nil)

(s/fdef get :args (s/cat :c (s/? ::gs/context) :r ::gs/resource :i ::gs/name) ret (s/nilable ::instance))
(defn get
  "Return the properties of the specified instance.

During plan, tracks all calls made, and sets up a dependency
relationship between instances. May return nil during planning, because
the instance might not exist yet.

During apply, returns the properties of the specified instance. Always returns non-nil."

  ([resource name]
   (assert *context* "get called outside of plan or apply")
   (let [r (resource/get-resource resource)]
     (cond
       *planning* (do
                    (swap! *deps* conj [resource name])
                    (let [i (-> (state/get-by-name (:state *context*) resource name))]
                      (when i
                        (let [props (-> (resource/get *context* resource (:identity i)) :properties)]
                          props))))
       *applying* (let [is (-> (state/get-by-name (:state *context*) resource name))
                        _ (assert is (<< "instance ~{resource} ~{name} not found"))
                        ir (resource/get *context* resource (:identity is))]
                    (assert ir (<< "instance ~{resource} ~{name} present in state but not resource"))
                    (or (:properties ir) {}))
       :else (assert false "get called outside of plan or apply"))))
  ([context resource name]
   (binding [*context* context
             *planning* true
             *deps* (atom #{})]
     (get resource name))))

(defn get-in
  "g/get + clojure.core/get-in. Calls get-in on the instance's properties"
  [resource name property-keys]
  (clojure.core/get-in (get resource name) property-keys))

(defn key [instance]
  (validate! ::gs/config-instance instance)
  [(:resource instance) (:name instance)])

(defn topo-sort
  "Returns a topological sort of the items in coll. f is a fn of an item in coll to its dependencies"
  [plan]
  {:pre [(s/valid? ::gs/plan plan)]
   :post [(do (when-not (= (count plan) (count %))
                (println "expected" (count plan) "got" (count %))) true)
          (= (count plan) (count %))
          (s/valid? ::gs/plan %)]}
  (let [items (group-by (fn [p]
                          (-> p :instance key)) plan)]
    (->> plan
         (map (fn [p]
                (let [i (:instance p)]
                  [(key i)  (::apply-dependencies i)])))
         (into {})
         (topo-sort/topo-sort)
         (mapv (fn [p]
                 (let [i (first (c/get items p))]
                   (when-not i
                     (assert false (<< "couldn't find dependency ~{p}")))
                   i))))))

(defn validate-dependencies! [desired-set]
  (doseq [d (vals desired-set)
          :let [deps (:dependencies d)]
          :when deps
          dep deps]
    (when-not (c/get-in desired-set dep)
      (throw (ex-info (format "dependency %s of % not found" dep d))))))

(defn plan-properties
  "Given a config instance, call the properties thunk if present, and update :dependencies"
  [i]
  (binding [*deps* (atom #{})
            *planning* true]
    (let [props (if (fn? (:properties i))
                  ((:properties i))
                  (:properties i))
          props (merge props (select-keys i [:identity]))
          i (assoc i ::plan-properties props)
          dependencies (or (:dependencies i) #{})
          i (update-in i [:dependencies] (fnil set/union #{}) @*deps*)]
      i)))

(defn with-apply-dependencies
  "Given a plan, assoc ::apply-dependencies to each plan step,
  containing only the dependencies that will run in this plan (rather
  than all dependencies the instance requires)"
  [plan]
  {:pre [(s/valid? ::gs/plan plan)]
   :post [(s/valid? ::gs/plan %)]}
  (let [mutating (set (map (fn [p]
                             (-> p :instance key)) plan))]
    (->>
     plan
     (map (fn [p]
            (let [p-deps (set/intersection (or (-> p :instance :dependencies) #{}) mutating)]
              (assoc-in p [:instance ::apply-dependencies] p-deps)))))))

(defn assert-resources-registered [desired]
  (assert (every? (fn [d]
                    (resource/get-resource (:resource d))
                    true) desired)))

(defn validate-name-uniqueness [desired]
  (let [dupes (duplicates (map key desired))]
    (when (seq dupes)
      (assert false (format "names must be distinct: duplicated: %s" dupes)))))

(defrecord PlanOperation [action instance])

(defmethod clojure.core/print-method PlanOperation
  [plan ^java.io.Writer writer]
  (.write writer (str "#plan" "[" (:action plan) " " (key (:instance plan)) " " (if (:diff plan)
                                                                                  (str (:remove (:diff plan)) "=>" (:add (:diff plan)))
                                                                                  "") "]")))

(defn maybe-update
  "Given a config instance and existing instance, return an Update operation, if necessary"
  [config-instance existing-instance]
  (let [[in-config in-existing common] (data/diff (::plan-properties config-instance) (select-keys (:properties existing-instance) (keys (::plan-properties config-instance))))]
    (if (or (seq in-config) (seq in-existing))
      (if (resource/update? (:resource config-instance))
        (let [op (map->PlanOperation {:action :update
                                      :instance (assoc config-instance :identity (:identity existing-instance))
                                      :diff {:remove in-existing
                                             :add in-config}})]
          (println "update:" (:resource config-instance) (:name config-instance) in-existing "=>" in-config)
          (validate! ::gs/plan-operation op "plan update operation does not conform")
          op)
        (do
          (println "resource" (key config-instance) "modified, no update operation, ignoring" "desired:" (seq in-config) "existing:" (seq in-existing))
          nil))
      nil)))

(defn plan
  "Returns a seq of maps of proposed actions to take to make resources
  reflect the desired state specified in config. If the changes are
  desired, pass the returned plan to g/apply!"
  [context config]
  (resource/load-providers)
  (validate! ::gs/context context)
  (validate! ::gs/configuration config)
  (binding [*context* context]
    (let [{:keys [state]} context
          managed (state/list state)
          desired (->> config
                       (map plan-properties))
          all-dependencies (apply set/union (map :dependencies desired))
          _ (assert-resources-registered desired)
          _ (validate-name-uniqueness desired)
          desired-set (group-by key desired)
          managed-set (group-by key managed)
          _ (validate-dependencies! desired-set)
          plan-remove (reduce (fn [remove i]
                                (if (and (not (c/get desired-set (key i))))
                                  (let [op (map->PlanOperation {:action :delete
                                                                :instance i})]
                                    (validate! ::gs/plan-operation op "plan remove operation does not conform")
                                    (conj remove op))
                                  remove)) [] managed)
          _ (validate! ::gs/plan plan-remove)
          plan-add (reduce (fn [add i]
                             (if (and (not (c/get managed-set (key i))))
                               (let [op (map->PlanOperation {:action :create
                                                             :instance i})]
                                 (validate! ::gs/plan-operation op "plan add operation does not conform")
                                 (conj add op))
                               add)) [] desired)
          _ (validate! ::gs/plan plan-add)
          plan-update (->> desired
                           (map (fn [config-instance]
                                  (when-let [existing (first (c/get managed-set (key config-instance)))]
                                    (maybe-update config existing))))
                           (filter identity))
          _ (validate! ::gs/plan plan-update)

          plan (concat (-> plan-remove with-apply-dependencies topo-sort)
                       (->> (concat plan-add plan-update) with-apply-dependencies topo-sort reverse))]
      (validate! ::gs/plan plan)
      plan)))

(defn pprint-plan [plan]
  (println "Plan:")
  (doseq [p plan]
    (println (:action p)
             (-> p :instance :resource)
             (-> p :instance :name)
             (-> p :instance :identity)
             (when (:diff p)
               (str (:remove (:diff p)) "=>" (:add (:diff p)))))))

(defn state? [x]
  (satisfies? state/State x))

(s/fdef import-instance :args (s/cat :c ::gs/context :r ::gs/resource :i ::gs/identity :n ::gs/name))
(defn import-instance
  "load a single _existing_ instance into the state DB. Name should map
  to a :name in the config"
  [context resource identity name]
  (let [r (resource/get-resource resource)
        i (resource/get context resource identity)
        _ (assert i)
        i (assoc i :name name)
        state (:state context)]
    (state/ensure state i)))

(s/fdef create :args (s/cat :c ::gs/context :r ::gs/resource :i ::gs/config-instance))
(defn create [context resource instance]
  (let [inst (-> (resource/create context resource instance)
                 (merge (select-keys instance [:name :dependencies])))]
    (state/create (:state context) inst)))

(s/fdef delete :args (s/cat :c ::gs/context :r ::gs/resource :i ::gs/identity))
(defn delete [context resource identity]
  (resource/delete context resource identity)
  (state/delete (:state context) identity))

(s/fdef update :args (s/cat :c ::gs/context :r ::gs/resource :i ::gs/managed-instance))
(defn update [context resource instance]
  (let [updated (-> (resource/update context resource instance)
                    (merge (select-keys instance [:name :dependencies])))]
    (state/update (:state context) updated)))

(defn maybe-resolve-properties-thunk [instance]
  (update-in instance [:properties] (fn [p]
                                      (if (fn? p)
                                        (p)
                                        p))))

(s/fdef apply! :args (s/cat :c ::gs/context :p ::gs/plan))
(defn apply! [context plan]
  "Takes a set of actions returned by g/plan"
  (validate! ::gs/context context)
  (validate! ::gs/plan plan)
  (binding [*applying* true
            *context* context]
    (doseq [p plan]
      (let [{:keys [state]} context
            {:keys [action instance]} p
            {:keys [identity resource name]} instance
            instance (:instance p)
            instance (maybe-resolve-properties-thunk instance)]
        (println "applying" action resource name)
        (condp = action
          :create (create context resource instance)
          :delete (delete context resource identity)
          :update (update context resource instance))))))

(defn list-unmanaged
  "Returns a list of all existing, unamanged resources"
  [context]
  {:pre [(validate! ::context context)]}
  (let [state (:state context)
        existing (->> (resource/list-resources)
                      vals
                      (map (fn [resource]
                             (let [lf (:list resource)
                                   instances (lf context)]
                               (map (fn [inst]
                                      (-> inst
                                          (merge {:resource (:resource resource)}))) instances))))
                      (c/apply concat))
        managed (->> (map (fn [e]
                            (let [ei (:instance e)
                                  id (:identity e)
                                  _ (assert id)
                                  stored (state/get state id)]
                              (if (and stored (:name stored))
                                (assoc e :name (:name stored))
                                e))) existing)
                     (filter :name))
        existing-set (set (map (fn [i]
                                 (select-keys i [:resource :identity])) existing))
        managed-set (set (map (fn [i]
                                (select-keys i [:resource :identity])) managed))
        unmanaged (set/difference existing-set managed-set)]
    unmanaged))

(defn purge!
  "Attempt to delete everything in state"
  [context]
  (doseq [i (-> context :state state/list)]
    (println "deleting" (:resource i) (:identity i))
    (delete context (:resource i) (:identity i))))

(defn garbage-collect
  "Remove from state everything in state that is not present in resource"
  [context]
  (->> context
       :state
       state/list
       (group-by key)
       (map (fn [[key insts]]
              (remove (fn [i]
                        (resource/get context (:resource i) (:identity i))) insts)))
       (filter (fn [is]
                 (seq is)))
       (apply concat)
       (map (fn [i]
              (println "delete" (dissoc i :properties))
              (state/delete (:state context) (:identity i))))
       (dorun)))

(instrument-ns)
