(ns genesis.resource
  (:refer-clojure :exclude [get list update])
  (:require [clojure.spec.alpha :as s]
            [clojure.core :as c]
            [genesis.specs :as gs]
            [genesis.util :refer [validate! instrument-ns]]
            [clojure.tools.namespace :as ns]))

(defonce resources (atom {}))

(s/def ::defresource (s/keys :req-un [:resource/create
                                      :resource/delete
                                      :resource/get
                                      :resource/list]
                             :opt-un [:resource/update]))

(defn create-resource [key args]
  (validate! ::defresource args "resource does not conform")
  (swap! resources assoc key (assoc args :resource key))
  nil)

(defn load-providers- []
  (->> (ns/find-namespaces-on-classpath)
       (filter (fn [n] (re-find #"^genesis\.provider" (name n))))
       (map (fn [n]
              (require n)))
       (dorun)))

(def load-providers-delay
  (delay (load-providers-)))

(def load-providers (fn [] @load-providers-delay))

(s/fdef get-resource :args (s/cat :r qualified-keyword?) :ret ::defresource)
(defn get-resource [resource]
  {:pre [(validate! qualified-keyword? resource)]
   :post [(do (when-not % (println "resource " resource " not loaded")) true) %]}
  (load-providers)
  (c/get @resources resource))

(defn list-resources []
  (load-providers)
  (seq @resources))

(defn list [context resource]
  {:pre [(validate! qualified-keyword? resource)]
   :post [(validate! ::gs/existing-instances %)]}
  ((:list (get-resource resource)) context))

(s/fdef get :args (s/cat :context :gs/context :r ::gs/resource :i ::gs/identity))
(defn get [context resource identity]
  {:pre [(validate! qualified-keyword? resource)
         (validate! ::gs/identity identity)]
   :post [(validate! (s/nilable ::gs/existing-instance) %)]}
  (when-let [i ((:get (get-resource resource)) context identity)]
    (assoc i :resource resource)))

(s/fdef create :args (s/cat :c ::gs/context :r ::gs/resource :i ::gs/config-instance) :ret ::gs/managed-instance)
(defn create [context resource instance]
  ((:create (get-resource resource)) context instance))

(s/fdef delete :args (s/cat :c ::gs/context :r ::gs/resource :identity ::gs/identity))
(defn delete [context resource identity]
  ((:delete (get-resource resource)) context identity))

(s/fdef update :args (s/cat :c ::gs/context :resource ::gs/resource :i ::gs/managed-instance) :ret ::gs/managed-instance)
(defn update [context resource instance]
  (-> ((:update (get-resource resource)) context instance)
      (assoc :resource resource)))

(s/fdef update? :args (s/cat :resource qualified-keyword?) :ret boolean?)
(defn update?
  "True if this resource supports update"
  [resource]
  (validate! qualified-keyword? resource)
  (boolean (:update (get-resource resource))))

(defn get-fn [action]
  {:post [%]}
  (condp = action
    :create create
    :delete delete
    :list list
    :get get))

(instrument-ns)
