(ns genesis.providers.aws.route53.hosted-zone
  (:require [amazonica.aws.route53 :as route53]
            [clojure.core.strint :refer (<<)]
            [clojure.java.shell :as sh]
            [clojure.set :as set]
            [clojure.spec.alpha :as s]
            [genesis.core :as g :refer [defresource]]
            [genesis.providers.aws.ec2 :as gec2]
            [genesis.util :refer [validate!]]))

(s/def ::vpc-id string?)
(s/def ::vpc (s/keys :req-un [::vpc-id]))
(s/def ::name string?)
(s/def ::private boolean?)
(s/def ::config (s/keys :opt-un [::private]))
(s/def ::hosted-zone (s/keys :opt-un [::vpc ::config]))

(defn normalize [hz]
  (-> hz
      (update-in [:config] set/rename-keys {:private-zone :private})
      (dissoc :caller-reference)))

(defn create-hosted-zone [context hosted-zone]
  (when (-> hosted-zone :config :private)
    (assert (-> hosted-zone :vpc :vpc-id) ":vpc {:vpc-id} is required when :private"))
  (validate! ::hosted-zone hosted-zone)
  (let [creds (-> context :aws :creds)
        z (-> (route53/create-hosted-zone creds (assoc hosted-zone
                                                       :caller-reference (str (System/currentTimeMillis))))
              :hosted-zone)]
    {:resource :route53/hosted-zone
     :identity (:id z)
     :properties (normalize z)}))

(defn list-hosted-zones [context]
  (->> (route53/list-hosted-zones (-> context :aws :creds) {})
       :hosted-zones
       (map (fn [z]
              (let [z (normalize z)]
                {:identity (:id z)
                 :properties z})))))

(def get-hosted-zone (gec2/get-by-identity list-hosted-zones))

(def delete-hosted-zone (gec2/delete-by-identity {:get-fn get-hosted-zone
                                                  :delete-fn route53/delete-hosted-zone
                                                  :id-key :id}))

(defresource :route53/hosted-zone {:create create-hosted-zone
                                   :list list-hosted-zones
                                   :get get-hosted-zone
                                   :delete delete-hosted-zone})
