# genesis

Terraform, clojure-style.

![Genesis Device](https://i.stack.imgur.com/zVb5Y.jpg)

Genesis a declarative tool for configuring infrastructure. It is a
Clojure library, not a standalone CLI tool.

## Motivation

- Terraform's .tf language is artifically constraining
- Terraform's standard library and datastructures are limited
- Instead, use a full-power programming language to specify configuration
- JSON-in-strings and XML-in-strings are harder to read and validate. Write clojure data literals and serialize as necesssary
- complex projects require terraforming in multiple steps, or across multiple AWS environments, which necessitates glue scripts
- in large projects, sharing secrets and env vars between terraform and the deploy process is annoying

## Design Choices

- genesis is a library, not a CLI tool
- it is expected you will use namespaces, functions, standard clojure factoring tools
- use functions to generate declarative data
- leverage existing provider APIs (e.g. amazonica) as much as possible

# Terminology

- *resource* : a set of APIs that allows create/update/delete new things of the same type, i.e. ec2 instances, s3 buckets or VPCs
- *provider*: a set of related APIs that share endpoints or credentials, i.e. AWS or GCP
- *instance* : a single object created by a resource, such as an ec2 instance
- *name*: the human-given name for an instance. Stays the same even if an instance is deleted and re-created. Names are unique per-resource
- *identity* : the API-given name used to identify an instance, i.e. i-abcdef. Globally unique
- *configuration* : a seq of clojure maps that specify the desired state of all resources managed by genesis
- *plan* : a proposed set of actions to take to make the desired state reality
- *state*: a DB tracking all instances genesis has created or is managing. state is responsible for mapping names to identities.

## Config

A configuration is a seq of maps. Each map represents a single
instance that we want to exist. Each must contain `:resource` `:name`,
and optionally, `:properties`

```clojure
(def config
  [{:resource :s3/bucket
    :name "genesis-test"}
   {:resource :ec2/vpc
    :name (format "VPC-%s" "dev")
    :properties {:cidr-block "10.64.0.0/16"}}])
```

The configuration specifies the desired set of instances. Every
instance has a `:name`, which is human-given.

Properties is a map of extra configuration, passed to the resource
API.

Because the config is plain clojure data, using standard clojure
factoring tools such as vars, functions, namespaces is encouraged.

For example, your config might look like

```clojure
(def config
  (concat dev-config (env-config "blue") (env-config "green"))
```

Again, configs are standard clojure, so any normal clojure technique
is useful here. Write functions that return config instances, split
them by namespaces, etc.

## state

Genesis uses a database to track all instances Genesis has created or
manages. When instances are created, the provider API returns
machine-generated ids, e.g. `i-abcdef`. The DB's primary purpose is
mapping `names` to `identities`.

Genesis uses external state rather than storing the mapping in
provider naming/tagging APIs because not all provider APIs provide
reliable ways of tracking names on all instances. For example in AWS,
many APIs supporting tagging, but not 100% of them. Without 100%
consistent tagging, it is not possible to determine whether genesis
created a given instance, and if genesis is responsible for cleaning
it up when it is removed from the config.

State is an instance of the `genesis.state/State` protocol. The choice
of how to store state is pluggable. Currently the only implementation
is a flat file in `genesis.state.fs`, but could easily be extended to
an S3 file or a SQL database.

## Plan and Apply

```clojure
(def context {:state (genesis.state.fs/new {:path ".genesis-state"})
              :aws {:creds env-creds}})
(g/plan context config)
```

Context is a map, containing `:state` and optional provider-specific
configuration. Context is passed to all resource API calls.

Plan returns a seq of maps, showing the actions genesis is proposing
to make, it might look like:

```clojure
(g/plan context config)
=>
[{:resource :ec2/vpc, :name "VPC-dev", :action :create}...]
```

Each map in the returned plan is the proposed action to take on a
single instance: `:create`, `:delete`, `:update`, etc.

`plan` does no modification, and may be run any number of times
safely. If the proposed set of actions is agreeable, pass the returned
plan to `g/apply!`

Instances specified in config but not existing are created. Instances
that exist and are present in the state DB but not specified in config
are deleted. Instances that exist in both are diff'd. If there are
differences, the resource's `update` API is used. If `update`ing a
particular property is not supported, the existing instance is
deleted, and a new instance is created.

Pass the returned plan from `g/plan` to `g/apply!`. Apply executes
the proposed actions, and updates the state DB.

## Properties

```clojure
{:resource :autoscaling/launch-config
 :name "genesis"
 :properties {:image-id "ami-abcdef"
              :instance-type "t2.medium"}}
```

Properties may be a map or a thunk (fn of no arguments) that returns a
map. The properties data is passed (mostly) as-is into the relevant
provider API. Properties on existing instances are diff'd between
repeated runs.

If properties are a thunk, the fn will be called during both
plan and apply.

```clojure
{:resource :route53/hosted-zone
 :name "genesis"
 :properties (fn [] {:name "dev.genesis.io"
                     :vpc {:vpc-id (:vpc-id (g/get :ec2/vpc "VPC-dev"))}})}
```

`g/get` is a function that returns the properties for instance with
the specified resource and name. In this example, we are creating a
hosted zone, and passing a VPC id.

During the planning phase, `g/get` tracks all calls made, and sets up
a dependency relationship between instances. Plan actions are
topologically sorted, so the action for an instance only happens after
its dependencies actions have finished.

Note that you should always use `g/get` to retreive dependencies, even
when you know the hard-coded value, for example:

```clojure
{:resource :autoscaling/launch-config
 :name "foo"
 :properties {:instance-type "t2.medium"}}

{:resource :autoscaling/group
 :name "foo"
 :properties (fn [] {:launch-configuration-name (-> (g/get :autoscaling/launch-config "foo") :launch-configuration-name)})}
```

Here, we know the launch-config is going to be named "foo", but `get`
is still recommended so genesis knows not to create the autoscaling
group before the launch configuration, nor delete the launch config
before deleting the ASG.

Property thunks should be idempotent, but feel free to call provider
fns to generate properties. The context passed to `g/plan` &
`g/apply!` are available at `g/*context*`.

# Providers

## AWS

Genesis uses amazonica for all AWS APIs. The context map should
contain `{:aws {:creds <creds>}}`. Creds can be in [any format
amazonica
accepts](https://github.com/mcohen01/amazonica#authentication).  In
general, genesis passes properties into the amazonica API with minimal
modification. Some APIs do light modification such as
base64-encode/decode, and url-encode/decode to make the config easier
to read and modify.

## License

Copyright © 2017 Griffin Financial Technology Ltd.

Distributed under the Eclipse Public License either version 1.0 or (at
your option) any later version.
