(ns genesis.state.fs
  (:require [clojure.data :refer (diff)]
            [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.spec.alpha :as s]
            [clojure.set :as set]
            [genesis.state :as state]
            [genesis.specs :as gs]
            [genesis.core :as g]
            [genesis.util :refer [validate!]]))


;;(defn reindex! )

(s/def ::index-name (s/map-of ::gs/key ::gs/existing-instance))
(s/def ::index-identity (s/map-of ::gs/identity ::gs/existing-instance))
(s/def ::disk-db (s/keys :req-un [::index-identity]))
(s/def ::db (s/keys :req-un [::index-name ::index-identity]))

(defn index! [db]
  {:pre [(validate! ::disk-db @db)]
   :post [(validate! ::db @db)
          (= (-> @db :index-identity vals count)
             (-> @db :index-name vals count))]}
  (dosync
   (let [index-name (->> @db
                         :index-identity
                         vals
                         (filter :name)
                         (map (fn [inst]
                                [(g/key inst) inst]))
                         (into {}))]
     (alter db assoc :index-name index-name))))

(defmethod print-dup org.joda.time.DateTime
  [o writer]
  (.write writer (format "#inst \"%s\"" o)))

(defmethod print-method org.joda.time.DateTime
  [o writer]
  (.write writer (format "#inst \"%s\"" o)))

(defn save! [path db]
  (println "saving:" path)
  (spit (io/file path) (pr-str (select-keys @db [:index-identity]))))

(defn empty-db []
  (ref {:index-identity {}
        :index-name {}}))

(defn load-db [path]
  {:post [(validate! ::db (deref %))]}
  (if (.exists (io/file path))
    (let [db (ref (edn/read-string (slurp path)))]
      (index! db)
      db)
    (empty-db)))

(defrecord FSState [path db]
  state/State
  (init- [this]
    (edn/read-string (slurp path)))
  (create- [this instance]
    (dosync
     (alter db assoc-in [:index-identity (:identity instance)] instance)
     (index! db)
     (save! path db)))
  (list- [this]
    (->> (get-in @db [:index-identity])
         vals))
  (get- [this identity]
    (get-in @db [:index-identity identity]))
  (get-by-name- [this resource name]
    (get-in @db [:index-name [resource name]]))
  (delete- [this identity]
    (dosync
     (alter db update-in [:index-identity] dissoc identity))
    (save! path db))
  (update- [this instance]
    (dosync
     (alter db assoc-in [:index-identity (:identity instance)] instance)
     (index! db))
    (save! path db)))

(defn new [& [{:keys [path]
               :or {path ".genesis-state"}}]]
  (map->FSState {:path path
                 :db (load-db path)}))
