(ns genesis.providers.aws.ec2.vpc
  (:require [amazonica.aws.ec2 :as ec2]
            [circle.wait-for :refer (wait-for)]
            [clj-time.core :as time]
            [genesis.providers.aws.ec2 :as gec2]
            [clojure.set :as set]
            [clojure.spec.alpha :as s]
            [clojure.data :as data]
            [genesis.core :as g :refer [defresource]]
            [genesis.util :refer [base64-encode base64-decode validate! unwrap!]]))

(def list-vpcs (let [f (gec2/list-with-identity ec2/describe-vpcs {:resource :ec2/vpc
                                                                   :id-key :vpc-id})]
                 (fn [context]
                   (->> (f context)
                        (map (fn [vpc]
                               (->
                                vpc
                                (assoc-in [:properties :amazon-provided-ipv6-cidr-block] (-> vpc :properties :ipv6cidr-block-association-set seq boolean))
                                (update-in [:properties] set/rename-keys {:ipv6cidr-block-association-set :ipv6-cidr-block-association-set}))))))))

(s/def ::cidr-block string?)
(s/def ::amazon-provided-ipv6-cidr-block boolean?)

(s/def ::vpc (s/keys :req-un [::cidr-block]
                     :opt-un [::amazon-provided-ipv6-cidr-block]))

(def create-vpc (gec2/create-with-identity ec2/create-vpc {:spec ::vpc
                                                           :id-key :vpc-id}))

(def get-vpc (gec2/get-by-identity list-vpcs))

(defn update-vpc [context identity properties]
  (let [creds (-> context :aws :creds)
        inst (get-vpc context identity)]
    (assert inst)
    (when (and (-> inst :properties :ipv6-cidr-block-association-set seq nil?)
               (:amazon-provided-ipv6-cidr-block properties))
      (ec2/associate-vpc-cidr-block creds {:vpc-id (:identity inst)
                                           :amazon-provided-ipv6-cidr-block (:amazon-provided-ipv6-cidr-block properties)}))
    (get-vpc context identity)
    ;; (when (and (-> inst :properties :ipv6-cidr-block-association-set seq)
    ;;            (not (:amazon-provided-ipv6-cidr-block properties)))
    ;;   (ec2/disassociate-vpc-cidr-block creds {:vpc-id (:identity inst)
    ;;                                           :amazon-provided-ipv6-cidr-block (:amazon-provided-ipv6-cidr-block properties)}))
    ))

(defn delete-vpc-subnets [creds vpc]
  (let [subnets (->> (ec2/describe-subnets creds {:filters [{:name "vpc-id" :value (:vpc-id vpc)}]})
                     :subnets)]
    (doseq [s subnets]
      (println "deleting vpc" (:vpc-id vpc) "subnet" (:subnet-id s))
      (ec2/delete-subnet creds {:subnet-id (:subnet-id s)}))))

(defn delete-vpc-internet-gateways [creds vpc]
  (assert (:vpc-id vpc))
  (let [igs (->> (ec2/describe-internet-gateways creds {:filters [{:key "attachment.vpc-id" :value (:vpc-id vpc)}]})
                 :internet-gateways)]
    (doseq [ig igs]
      (println "detaching vpc" (:vpc-id vpc) "internet gateway" (:internet-gateway-id ig))
      (ec2/detach-internet-gateway creds {:internet-gateway-id (:internet-gateway-id ig)
                                          :vpc-id (:vpc-id vpc)})
      (println "deleting vpc" (:vpc-id vpc) "internet gateway" (:internet-gateway-id ig))
      (ec2/delete-internet-gateway creds {:internet-gateway-id (:internet-gateway-id ig)}))))

(defn delete-vpc-subnets [creds vpc]
  (let [subnets (->> (ec2/describe-subnets creds {:filters [{:key "vpc-id" :value (:vpc-id vpc)}]})
                 :subnets)]
    (doseq [subnet subnets]
      (println "deleting vpc" (:vpc-id vpc) "subnet" (:subnet-id subnet))
      (ec2/delete-subnet creds (select-keys subnet [:subnet-id])))))

(defn delete-route-table-routes [creds vpc route-table]
  (doseq [r (:routes route-table)
          :when (not= (:origin r) "CreateRouteTable")]
    (println "deleting vpc" (:vpc-id vpc) "route-table" (:route-table-id route-table) "route" r)
    (ec2/delete-route creds (merge
                             (select-keys route-table [:route-table-id])
                             (select-keys r [:destination-cidr-block])))))

(defn route-table-main? [route-table]
  (some (fn [a]
          (:main a)) (:associations route-table)))

(defn delete-vpc-route-tables [creds vpc]
  (let [rts (->> (ec2/describe-route-tables creds)
                 :route-tables
                 (filter (fn [rt]
                           (and (= (:vpc-id vpc) (:vpc-id rt))
                                (not (route-table-main? rt))))))]
    (doseq [rt rts]
      (delete-route-table-routes creds vpc rt)
      (println "deleting vpc" (:vpc-id vpc) "route-table" (:route-table-id rt))
      (ec2/delete-route-table creds (select-keys rt [:route-table-id])))))

(defn delete-vpc-security-groups [creds vpc]
  (let [sgs (->> (ec2/describe-security-groups creds)
                 :security-groups
                 (filter (fn [sg]
                           (and (= (:vpc-id vpc) (:vpc-id sg))
                                (not= "default" (:group-name sg))))))]
    (doseq [sg sgs]
      (println "deleting vpc" (:vpc-id vpc) "security-group" (:group-id sg))
      (ec2/delete-security-group creds (select-keys sg [:group-id])))))

(defn delete-vpc [context identity]
  {:pre [(string? identity)]}
  (let [creds (-> context :aws :creds)
        vpc (:properties (get-vpc context identity))]
    (assert vpc)
    (assert (:vpc-id vpc))
    (delete-vpc-security-groups creds vpc)
    (delete-vpc-subnets creds vpc)
    (delete-vpc-route-tables creds vpc)
    (delete-vpc-internet-gateways creds vpc)
    (ec2/delete-vpc creds (select-keys vpc [:vpc-id]))))

(defresource :ec2/vpc {:list list-vpcs
                       :get get-vpc
                       :create create-vpc
                       :delete delete-vpc
                       :update update-vpc})

(defn all-vpc-subnets [creds vpc-id]
  (->> (ec2/describe-subnets creds {:vpc-id vpc-id})
       :subnets
       (map :subnet-id)))


(s/def ::assign-ipv6-address-on-creation boolean?)
(s/def ::map-public-ip-on-launch boolean?)
(s/def ::route-table-id string?)

(s/def ::subnet (s/keys :req-un [::cidr-block
                                 ::vpc-id]
                        :opt-un [::assign-ipv6-address-on-creation
                                 ::map-public-ip-on-launch
                                 ::route-table-id]))

(defn subnet->route-table
  "Returns a map of {subnet-id route-table-info}, for subnets with non-default route-tables"
  [creds]
  (->> (ec2/describe-route-tables creds)
       :route-tables
       (map (fn [rt]
              (update-in rt [:associations] (fn [as]
                                              (map (fn [a]
                                                     (assoc a :route-table-id (:route-table-id rt))) as)))))
       (mapcat :associations)
       (remove :main)
       (filter :subnet-id)
       (map (fn [a]
              (assert (:route-table-id a))
              (when-not (:main a)
                [(:subnet-id a)
                 (select-keys a [:route-table-id :subnet-id :route-table-association-id])])))
       (filter identity)
       (into {})))

(def list-subnets (let [f (gec2/list-with-identity ec2/describe-subnets {:id-key :subnet-id})]
                    (fn [context]
                      (let [creds (-> context :aws :creds)
                            subnet-routes (subnet->route-table creds)]
                        (->> (f context)
                             (map (fn [inst]
                                    (let [subnet-id (-> inst :properties :subnet-id)]
                                      (-> inst
                                          (update-in [:properties :tags] gec2/tags->map)
                                          (cond->
                                              (contains? subnet-routes subnet-id) (assoc-in [:properties :route-table-id] (get-in subnet-routes [subnet-id :route-table-id]))))))))))))

(def get-subnet (gec2/get-by-identity list-subnets))


(defn ensure-subnet-route-table [context identity desired-route-table-id]
  (let [creds (-> context :aws :creds)
        subnet (get-subnet context identity)
        _ (assert subnet)
        subnet-routes (subnet->route-table creds)
        subnet-id (:identity subnet)
        actual-route-table-id (-> subnet :properties :route-table-id)]
    (cond
      (= actual-route-table-id desired-route-table-id) nil
      (and (not actual-route-table-id) desired-route-table-id) (ec2/associate-route-table creds {:subnet-id subnet-id
                                                                                                 :route-table-id desired-route-table-id})
      (and actual-route-table-id (not desired-route-table-id)) (let [association-id (get-in subnet-routes [subnet-id :route-table-association-id])]
                                                                 (assert association-id)
                                                                 (ec2/disassociate-route-table creds {:association-id association-id}))
      (not= actual-route-table-id desired-route-table-id) (let [association-id (get-in subnet-routes [subnet-id :route-table-association-id])]
                                                            (assert association-id)
                                                            (ec2/disassociate-route-table creds {:association-id association-id})
                                                            (ec2/associate-route-table creds {:subnet-id subnet-id
                                                                                              :route-table-id desired-route-table-id})))))

(def create-subnet (let [f (gec2/create-with-identity ec2/create-subnet {:spec ::subnet
                                                                         :id-key :subnet-id})]
                     (fn [context properties]
                       (let [creds (-> context :aws :creds)
                             inst (f context properties)
                             identity (:identity inst)
                             opts (select-keys properties [:assign-ipv6-address-on-creation :map-public-ip-on-launch])]
                         (when (:tags properties)
                           (gec2/tag-resource context identity (gec2/map->tags (:tags properties))))
                         (when (seq opts)
                           (ec2/modify-subnet-attribute creds (assoc opts :subnet-id identity)))
                         (ensure-subnet-route-table context identity (-> inst :properties :route-table-id))
                         (get-subnet context identity)))))

(def delete-subnet (gec2/delete-by-identity {:get-fn get-subnet
                                             :delete-fn ec2/delete-subnet
                                             :id-key :subnet-id}))

(defn update-subnet [context identity properties]
  (let [creds (-> context :aws :creds)
        subnet (get-subnet context identity)
        existing-tags (-> subnet :properties :tags)
        desired-tags (:tags properties)
        [to-add to-remove common] (data/diff desired-tags existing-tags)
        subnet-attrs (select-keys properties [:assign-ipv6-address-on-creation :map-public-ip-on-launch])]
    (doseq [a to-add]
      (gec2/tag-resource context identity (gec2/map->tags to-add)))
    (doseq [r to-remove]
      (gec2/untag-resource context identity (gec2/map->tags to-remove)))
    (when (seq subnet-attrs)
      (ec2/modify-subnet-attribute creds (assoc subnet-attrs :subnet-id identity)))
    (ensure-subnet-route-table context identity (:route-table-id properties))
    (get-subnet context identity)))

(defresource :vpc/subnet {:list list-subnets
                          :get get-subnet
                          :create create-subnet
                          :update update-subnet
                          :delete delete-subnet})

(defn main? [rtb]
  (->> rtb :properties :associations (some (fn [a]
                                             (:main a)))))

(def list-route-table (let [f (gec2/list-with-identity ec2/describe-route-tables {:id-key :route-table-id})]

                        (fn [context]
                          (->> (f context)
                               ;; an extra route-table gets created for free by VPCs. Don't show it in list, because it can't be deleted directly, and we didn't create it
                               (remove main?)))))

(def get-route-table (gec2/get-by-identity list-route-table))
(def delete-route-table (gec2/delete-by-identity {:get-fn get-route-table
                                                  :delete-fn ec2/delete-route-table
                                                  :id-key :route-table-id}))

(s/def ::route-table (s/keys :req-un []))

(def create-route-table (gec2/create-with-identity ec2/create-route-table {:spec ::route-table
                                                                           :id-key :route-table-id}))

(defresource :vpc/route-table {:list list-route-table
                               :get get-route-table
                               :create create-route-table
                               :delete delete-route-table})

(defn get-vpc-default-route-table [context vpc-id]
  ;; vpc-id can be nil when the vpc doesn't exist yet
  {:pre [(or (string? vpc-id) (nil? vpc-id))]
   :post [(if (string? vpc-id) % true)]}
  (let [creds (-> context :aws :creds)]
    (->> (ec2/describe-route-tables creds)
         :route-tables
         (filter (fn [rt]
                   (and (= vpc-id (:vpc-id rt))
                        (route-table-main? rt))))
         first)))

(def list-internet-gateways (gec2/list-with-identity ec2/describe-internet-gateways {:id-key :internet-gateway-id}))

(s/def ::internet-gateway (s/keys :req-un [::vpc-id]))

(def create-internet-gateway (let [f (gec2/create-with-identity ec2/create-internet-gateway {:spec ::internet-gateway
                                                                                             :id-key :internet-gateway-id})]
                               (fn [context properties]
                                 (let [creds (-> context :aws :creds)
                                       instance (f context properties)
                                       created-props (:properties instance)
                                       ig-id (:internet-gateway-id created-props)]
                                   (assert ig-id)
                                   (ec2/attach-internet-gateway creds {:vpc-id (:vpc-id properties)
                                                                       :internet-gateway-id ig-id})
                                   instance))))

(def get-internet-gateway (gec2/get-by-identity list-internet-gateways))

(def delete-internet-gateway (gec2/delete-by-identity {:get-fn get-internet-gateway
                                                       :delete-fn ec2/delete-internet-gateway
                                                       :id-key :internet-gateway-id}))

(defresource :vpc/internet-gateway {:list list-internet-gateways
                                    :get get-internet-gateway
                                    :create create-internet-gateway
                                    :delete delete-internet-gateway})


(def list-nat-gateways (gec2/list-with-identity #(ec2/describe-nat-gateways % {}) {:id-key :nat-gateway-id}))

(def get-nat-gateway- (gec2/get-by-identity list-nat-gateways))

(defn wait-for-not-pending [context identity]
  (let [instance (get-nat-gateway- context identity)]
    (if (-> instance :properties :state (= "pending"))
      (wait-for {:timeout (time/minutes 10)
                 :sleep (time/seconds 5)}
                #(do
                   (println "waiting for nat-gateway " (:identity instance) "to not be pending")
                   (-> (get-nat-gateway- context identity) :properties :state (= "pending"))))
      instance)))

(defn get-nat-gateway [context identity]

  (wait-for-not-pending context identity))

(s/def ::allocation-id string?)
(s/def ::nat-gateway (s/keys :req-un [::allocation-id ::subnet-id]))

(def create-nat-gateway (let [f (gec2/create-with-identity ec2/create-nat-gateway {:spec ::nat-gateway
                                                                                   :id-key :nat-gateway-id})]
                          (fn [context properties]
                            (let [i (f context properties)]
                              (assert (not= "failed" (-> i :properties :state)))
                              i))))

(def delete-nat-gateway (gec2/delete-by-identity {:get-fn get-nat-gateway
                                                  :delete-fn ec2/delete-nat-gateway
                                                  :id-key :nat-gateway-id}))

(defresource :vpc/nat-gateway {:list list-nat-gateways
                               :get get-nat-gateway
                               :create create-nat-gateway
                               :create-spec ::nat-gateway
                               :delete delete-nat-gateway})



(s/def ::route-table (s/keys :req-un [::vpc-id]))

(def create-route-table (gec2/create-with-identity ec2/create-route-table {:spec ::route-table
                                                                           :id-key :route-table-id}))


(def list-route-tables
  (fn [context]
    (let [f (gec2/list-with-identity ec2/describe-route-tables {:id-key :route-table-id})]
      (->> (f context)
           (remove (fn [rt]
                     (-> rt :properties :main)))))))

(def get-route-table (gec2/get-by-identity list-route-tables))

(def delete-route-table (gec2/delete-by-identity {:get-fn get-route-table
                                                  :delete-fn ec2/delete-route-table
                                                  :id-key :route-table-id}))

(defresource :vpc/route-table {:list list-route-tables
                               :get get-route-table
                               :create create-route-table
                               :delete delete-route-table})


(s/def ::route-table-id string?)
(s/def ::destination-cidr-block string?)
(s/def ::destination-ipv6-cidr-block string?)
(s/def ::egress-only-internet-gateway-id string?)
(s/def ::gateway-id string?)
(s/def ::nat-gateway-id string?)
(s/def ::instance-id string?)
(s/def ::network-interface-id string?)
(s/def ::vpc-peering-connection-id string?)

(s/def ::route (s/keys :req-un [::route-table-id
                                (or
                                 ::egress-only-internet-gateway-id
                                 ::gateway-id
                                 ::nat-gateway-id
                                 ::instance-id
                                 ::network-interface-id
                                 ::vpc-peering-connection-id)
                                (or ::destination-cidr-block
                                    ::destination-ipv6-cidr-block)]))

(defn route-identity [properties]
  (let [route-table-id (:route-table-id properties)
        dest (or (:destination-cidr-block properties)
                 (:destination-ipv6-cidr-block properties))]
    (assert (string? route-table-id))
    (assert (string? dest) (format "no dest on: %s" properties))
    (str route-table-id " " dest)))

(defn create-route [context properties]
  (validate! ::route properties)
  (let [creds (-> context :aws :creds)
        _ (assert creds)
        _ (println "create-route:" properties)
        resp (ec2/create-route creds properties)]
    (if (:return resp)
      {:identity (route-identity properties)
       :properties properties}
      (throw (ex-info "create route failed:" resp)))))

(defn list-routes [context]
  (let [creds (-> context :aws :creds)]
    (->> (ec2/describe-route-tables creds)
         unwrap!
         (map (fn [rt]
                (map (fn [r]
                       (assoc r :route-table-id (:route-table-id rt))) (:routes rt))))
         (apply concat)
         (map (fn [r]
                (let [r (set/rename-keys r {:destination-ipv6cidr-block :destination-ipv6-cidr-block})]
                  {:identity (route-identity r)
                   :properties r}))))))

(def get-route (gec2/get-by-identity list-routes))

(defn delete-route [context identity]
  (let [creds (-> context :aws :creds)
        inst (get-route context identity)
        _ (assert inst)
        properties (:properties inst)
        _ (assert properties)]
    (ec2/delete-route creds (select-keys properties [:route-table-id :destination-cidr-block :destination-ipv6-cidr-block]))))

(defresource :vpc/route {:list list-routes
                         :get get-route
                         :create create-route
                         :delete delete-route})


(s/def ::egress-only-internet-gateway (s/keys :req-un [::vpc-id]))

(def create-egress-only-ig (gec2/create-with-identity ec2/create-egress-only-internet-gateway {:spec ::route-table
                                                                                               :id-key :egress-only-internet-gateway-id}))

(def list-egress-only-igs (gec2/list-with-identity ec2/describe-egress-only-internet-gateways {:id-key :egress-only-internet-gateway-id
                                                                                               :require-args? true}))

(def get-egress-only-ig (gec2/get-by-identity list-egress-only-igs))

(defn delete-egress-only-ig [context identity]
  (ec2/delete-egress-only-internet-gateway (-> context :aws :creds) {:egress-only-internet-gateway-id identity}))

(defresource :vpc/egress-only-internet-gateway {:list list-egress-only-igs
                                                :get get-egress-only-ig
                                                :create create-egress-only-ig
                                                :delete delete-egress-only-ig})
