(ns gov.nist.closmop
  (:require [clojure.pprint :refer (cl-format pprint)]
            [clojure.walk :refer (postwalk-replace)]))

(alias 'cm 'gov.nist.closmop)

;;; Purpose: Implement some Common Lisp Object System (CLOS) and Metaobject Protocol (MOP) functions.
;;;          Based on "Art of the Metaobject Protocol" by G. Kiczales et al.
;;;
;;; Note: The things defined in this file are sensitive to their place in the file because there is
;;;       some 'bootstrapping' going on. (Search for ";;; Bootstrapping..." in the code.)

;;; ToDo:
;;;        - Consider making it more clojure-idiomatic:
;;;                   (1) (defclass! ClassA [SuperclassA] {slot-1 { ...}}) <--- but think about slot order!
;;;                   (2) (:cm/direct-slots arg)
;;;                   (3) (persistent! obj)
;;;                       (Could implement as just (obj-map) ??? with (meta :type ::ClosObj)
;;;        - Make finalize-classes smarter (finalize-class cname)...
;;;             (cm/finalized? slot) mof.compat needed to run it.
;;;       - Move of-class information out of the slots and into the ClosmodObj.   
;;; DONE  - Make remove-method! work. See initialize-instance for mm-root-supertype.
;;;        - Does finalize-class care whether a superclass is forward-reference?
;;; DONE   - ??? Dump the whole "qualified symbol" thing??? Can still use (e.g. cm/class-name)
;;;               or ANYTHING! Just won't mess around with method names.
;;;          Maybe just give up on that for methods to start (leave class names alone???)
;;; DONE   - Eliminate ctries by using (atom {}) for +classes+, write add-class etc.
;;;        - Explore why with-slots now needs ns-qualified slots/vars.
;;;        - Fix inspect -- :dr doesn't work.
;;;        - Make +classes+ ^:private. 
;;;  DONE  - Fix print-method / print-object.
;;;  DONE  - Implement print-object as a closmop method; eliminate use of derive, if possible.
;;;        - Implement type hints in methods (as specializers?)
;;;  DONE  - Consider adding source-qualified initargs. 
;;;  DONE  - Consider making before qualifiers least-specific first
;;;  ????  - Replace 'cm/qualified symbols with ::namespace-qualified-keywords.
;;;  DONE  - Allocate instances with unbound slots.
;;;  DONE  - Implement initialize-instance as a method (testing qualifiers)
;;;  DONE  - Use {:pre [...] :post [...]} rather than throw/ex-info.
;;;  ????  - Fix problem of subclasses with slots named same as a superclass slot.
;;;  DONE  - implement initform
;;;  DONE  - Add :initarg to :initargs

(def ^:private +id+ (atom 0))

(defprotocol IClosmopObj
  (^clojure.lang.PersistentArrayMap obj-map [this])
   (^long obj-id [this])
   (^Object assocv [this slot val])
   (^Object getv   [this slot] [this slot error?]))
 
(deftype ClosmopObj
    [^java.lang.Long id
     ^:volatile-mutable ^clojure.lang.PersistentArrayMap map] ; volatile = readers see updates (because not in cache)
  IClosmopObj                                                           
  (obj-map [this] map)
  (obj-id [this] id)
  (assocv [this slot val]
    (locking this ; lock while reading/writing
      (set! map (assoc map slot val))
      val))
  (getv [this slot]
    (let [val (get map slot ::access-fail)]
      (case val
        ::access-fail    (throw (ex-info "getv: object does not possess slot" {:obj this :slot slot}))
        ::+unbound-val+ (throw (ex-info "getv on unbound slot" {:obj this :slot slot}))
        val)))
  (getv [this slot error?]
    (if error?
      (getv this slot)
      (get map slot ::access-fail))))

(defn mk-obj [m]
  (ClosmopObj. (swap! +id+ inc) m))

;;; Bootstrapping...
(def +classes+ (atom {'cm/std-object 
                      (mk-obj '{cm/of-class cm/std-class
                                cm/name cm/std-object
                                cm/direct-methods []
                                cm/cpl []
                                cm/effective-slots []
                                cm/direct-superclasses []
                                cm/finalized? false
                                cm/direct-slots {}})
                      'cm/std-class 
                      (mk-obj '{cm/name cm/std-class
                                cm/of-class cm/std-class
                                cm/cpl []
                                cm/effective-slots []
                                cm/direct-superclasses []
                                cm/finalized? false
                                cm/direct-methods []
                                cm/direct-slots
                                ;; These are the slot-descriptors, not values!
                                {cm/name {:order 0}, 
                                 cm/direct-superclasses {:initform [] :order 1},
                                 cm/direct-slots {:initform {} :order 2},
                                 cm/effective-slots {;:initargs (:std-class.effective-slots)
                                                     :initform {} :order 3},
                                 cm/cpl {:initform [] :order 4}, 
                                 cm/direct-methods {:initform [] :order 5}
                                 cm/finalized? {:initform false :order 6}}})}))

(def ^:private +gfuns+ (atom {}))
;;; ...End bootstrapping

(defn- assoc-atom
  "assoc on map in atom 'bash in place'. Return value is meaningless."
  [atom property val]
  (reset! atom (assoc @atom property val)))
    
(defmacro setv
  "Set an object property. This is written like CL (setf (slot-value obj slot) val) 
   to facilitate with-slots. BTW, it doesn't use define-set(f/v)-expander, currently." ; POD
  [getv-form val]
  `(let [obj# ~(second getv-form)
         prop# ~(nth getv-form 2)]
     (when-not (contains? (obj-map obj#) prop#)
       (throw (ex-info "Object does not define slot" {:obj obj# :prop prop#})))
     (assocv obj# prop# ~val)))

(defn qualified-symbol 
  "Return a symbol qualified by an alias of the current ns (if one exists) or the current ns name."
  ([s]
   (when (instance? clojure.lang.Named s)
     (if (namespace s) 
       s 
       (if-let [answer (some #(when (= (val %) *ns*) (key %)) (ns-aliases *ns*))]
         (symbol (str answer) (name s))
         (symbol (str (ns-name *ns*)) (name s))))))
  ([s ns]
   (when (instance? clojure.lang.Named s)
     (if-let [answer (some #(when (= (val %) ns) (key %)) (ns-aliases ns))]
       (symbol (str answer) (name s))
       (symbol (str (ns-name ns)) (name s))))))

(defn find-class 
  "Return a previously-defined class, produce an error, or if :error? false, return nil."
  [cname & {:keys [error?] :or {error? true}}]
  (if (= cname true)
    nil
    (let [real-name (qualified-symbol cname)]
      (or (get @+classes+ real-name)
          (when error? (throw (ex-info "Class not found" {:name real-name})))))))

(defn compute-initargs
  "Return a map {slot-name1 initform1...}"
  [cname]
  (let [obj (find-class cname)]
    (->
     (apply hash-map
            (mapcat (fn [[k v]] (list k (get v :initform ::+unbound-val+))) ; POD must this be (eval (get v...)?
                    (getv ^ClosmopObj obj 'cm/effective-slots)))
     (assoc 'cm/of-class cname))))

(defn boundp [obj slot]
  (not= ::+unbound-val+ (getv obj slot false)))

;(def debug-instances (atom {})) ; POD debugging
(defn- allocate-instance [cname] 
  (let [^ClosmopObj result (mk-obj (compute-initargs cname))]
;    (swap! debug-instances conj [@+id+ result])
    result))

;(defn get-id [id]
;  "Debug to get an instance by id."
;  (get @debug-instances id))

(defn class-of 
  "Return the class of OBJ. If OBJ is not a class instance, return true 
  (which is the class of everything else, for the time being at least)." ; POD fix this. 
  [^ClosmopObj obj]
  (if (instance? ClosmopObj obj)
    (getv obj 'cm/of-class)
    true))

(defn- find-slot-by-initarg 
  "Used by initialize-instance, this returns the slot associated with an initarg."
  [class initarg]
  (or (some (fn [slot] 
              (let [[_ v] slot]
                (when (some (fn [x] (= initarg x)) (:initargs v)) slot)))
            (getv (find-class class) 'cm/effective-slots))
      (throw (ex-info "Class does not possess the initarg." {:class class :initarg initarg}))))

;;; Bootstrapping....
(defn initialize-instance
  [instance initargs]
  (let [cname (class-of instance)
        iforms (apply hash-map
                      (mapcat (fn [[sname m]] 
                                (when-let [form (:initform m)]
                                  (list (first (:initargs m)) (eval form))))
                              (getv (find-class cname) 'cm/effective-slots)))]
    (doseq [[k v] (merge iforms initargs)]
      (assocv instance (first (find-slot-by-initarg cname k)) v)))
  instance)
;;; end bootstrapping

(defn- ensure-forward-ref-class 
  "Like ensure-class but just for FRCs. It exists to facilitate bootstrapping."
  [cname]
  {:pre [(symbol? cname)]}
  (let [c (allocate-instance 'cm/forward-ref-class)
        qname (qualified-symbol cname)]
    (setv (getv c 'cm/name) qname)
    (setv (getv c 'cm/cpl) [qname])
    (assoc-atom +classes+ qname c)
    c))

(defn- depth-first-preorder-superclasses* 
  [cname]
  (if-let [class (find-class cname :error? false)]
    (when (not= 'cm/forward-ref-class (class-of class))
      (conj (mapcat depth-first-preorder-superclasses* 
                    (getv (get @+classes+ cname) 'cm/direct-superclasses))
            cname))
    (do
      (ensure-forward-ref-class cname)
      nil)))

(defn compute-cpl
  "Compute the class precedence list of the argument class (a symbol naming the class)."
  [cname] ; POD distinct on lists seems to remove from end. Good.
  (if (= cname true)
    ()
    (if-let [^ClosmopObj class (find-class cname :error? false)]
      (let [result (into (vec (distinct (depth-first-preorder-superclasses* cname))) ['cm/std-object true])]
        (setv (getv class 'cm/cpl) result)
        result)
      (ensure-forward-ref-class cname))))

;;; Do not use getv; let frc and true through
;;; I tried to do this with sorted-map-by. Didn't work very well!
(defn sort-slots
  "Return a map of effective slots ordered, adding an :order value to each." 
  [cname slots]
  (if (not (empty? slots))
    (let [cpl (remove #(= true %) (getv (find-class cname) 'cm/cpl))
          order (map :name 
                     (sort
                      (fn [x y] ; x and y maps that include the slot name.
                        (let [sx (:source x)
                              sy (:source y)
                              ox (:order (get (getv (find-class sx) 'cm/direct-slots) (:name x)))
                              oy (:order (get (getv (find-class sy) 'cm/direct-slots) (:name y)))]
                          (if (= sx sy)
                            (< ox oy)
                            (< (.indexOf cpl sx)
                               (.indexOf cpl sy)))))
                      (map (fn [[k v]] (assoc v :name k)) slots)))]
      (loop [s slots
             o order]
        (if (empty? o)
          s
          (recur (assoc-in s [(first o) :order] (.indexOf order (first o)))
                 (rest o)))))
    {}))

(defn- compute-slots 
  "Compute the effective slots of the argument class (a symbol naming the class)."
  [cname]
  (when (not= cname true) ; Gather slots from subclasses in CPL and add :order and an initarg. 
    (let [result
          (sort-slots
           cname
           (apply hash-map 
                  (flatten
                   (mapcat (fn [source] (map (fn [[sname smap]]
                                               (list sname
                                                     (as-> smap ?map 
                                                       (assoc ?map :source source)
                                                       (update-in ?map [:initargs] conj 
                                                                  (keyword (str (name source) "." (name sname))))
                                                       (if (:initarg smap)
                                                         (update-in ?map [:initargs] conj (:initarg smap))
                                                         ?map))))
                                             ;; not getv let frc, true through
                                             (get (obj-map (or (find-class source :error? false) 
                                                               (cm/ensure-forward-ref-class source)))
                                                  'cm/direct-slots)))
                           (filter #(not= true %) (getv (find-class cname) 'cm/cpl))))))]
      (setv (getv (find-class cname) 'cm/effective-slots) result)
      result)))
        
;;; Some functions I never remember to use. Clojure 1.7 has experimental in-lining, BTW. 
(defn finalized? [cname]
  (if (= cname true)
    true
    (getv (find-class cname) 'cm/finalized?)))

(defn cpl [cname]
  (getv (find-class cname) 'cm/cpl))

(defn direct-slots [cname]
  (getv (find-class cname) 'cm/direct-slots))

(defn effective-slots [cname]
  (getv (find-class cname) 'cm/effective-slots))

(defn finalize-class
  [cname & {:keys [force]}]
  (when (and (not= cname true)
             (find-class cname :error? false)
             (or force (not (finalized? cname)))) 
    (when-let [class (find-class cname :error? false)]
      (setv (getv class 'cm/finalized?) true)
      (compute-cpl cname)
      (dorun
       (map
        (fn [[k v]]
          (when (some #(= % cname) (getv v 'cm/cpl))
            (map #(finalize-class % :force true)
                 (reverse (getv (find-class k) 'cm/cpl)))))
        @+classes+))
      (compute-slots cname)))
  true)

;;; POD only symbol cspec so far. 
(defn make-instance 
  "Create an instance. CSPEC can either be a symbol naming a class or a 
   list of symbols naming classes (the latter for programmatic multiple inheritance)."
  [cspec & {:keys [] :as initargs}]
  (when-let [c (find-class cspec)]
    (let [cspec* (getv c 'cm/name)]
      (finalize-class cspec*)
      (initialize-instance (allocate-instance cspec*) initargs))))

(defn- ensure-class-using-class [metaclass cname initargs]
  (let [qcname (qualified-symbol cname)]
    (assoc-atom +classes+ qcname
                (if (= metaclass 'cm/forward-ref-class) ; suppress finalization 
                  (ensure-forward-ref-class qcname)
                  (apply (partial make-instance (qualified-symbol metaclass))
                         (mapcat (fn [[k v]] (list k v))
                                 (as-> initargs ?args
                                   (assoc ?args :std-class.finalized? false) 
                                   (assoc ?args :std-class.direct-superclasses 
                                          (map qualified-symbol (:std-class.direct-superclasses ?args)))
                                   (assoc ?args :std-class.direct-slots 
                                          (apply hash-map (mapcat (fn [[k v]] (list (qualified-symbol k) v)) 
                                                                  (:std-class.direct-slots ?args))))
                                   (assoc ?args :std-class.name qcname))))))
    (finalize-class qcname)
    qcname))

(defn ensure-class 
  "The functional equivalent of defclass!."
  [cname & {:keys [] :as initargs}]
  (let [initargs
        (assoc initargs
               :std-class.direct-slots
               ;; Bug! https://groups.google.com/forum/#!topic/clojure/7xa7Yg0-htc
               (if (not (empty? (:std-class.direct-slots initargs)))
                 (map-indexed (fn [i slot]
                                [(first slot)
                                 (assoc (apply hash-map (rest slot)) :order i)])
                              (:std-class.direct-slots initargs))
                   ()))]
    (ensure-class-using-class
     (:metaclass initargs 'cm/std-class)
     cname
     (dissoc initargs :metaclass))))
        
;;; Bootstrapping...
(compute-cpl 'cm/std-object)  ; POD missing /id, /finalized? thus not finalize-class
(compute-slots 'cm/std-object)
(compute-cpl 'cm/std-class)
(compute-slots 'cm/std-class)
;;; ...End Bootstrapping.

(defmacro defclass!
  [cname superclasses slots & {:keys [metaclass] :or {metaclass 'cm/std-class}}]
  (let [qcname#  (qualified-symbol cname),
        qmclass# (qualified-symbol metaclass)]
    `(ensure-class '~qcname#
                   :metaclass '~qmclass#
                   :std-class.effective-slots []
                   :std-class.direct-superclasses
                   '~(if (empty? superclasses) () (doall (map qualified-symbol superclasses)))
                   :std-class.direct-slots '~slots)))

(defclass! forward-ref-class ()
  ((name)
   (cpl :initform [])
   (effective-slots :initform [])
   (direct-superclasses :initform [])
   (finalized? :initform true)))

;;;======================= Methods ===========================
(def ^{:dynamic true :private true} *pq* ())        ; POD use atom?
(def ^{:dynamic true :private true} *pq-state* ())  ; POD use atom?

(defn- parse-lambda
  "Analyze the lambda-list, returning a map of keywords, keyword argument specs, 
   variable names, variable names and specializers, just specializers, the rest variable,
   optionals, and (maybe) allow-other-keys" 
  [ll]
  (binding [*pq* (apply list ll), ; POD because I first wrote it thinking of lambda list, not lambda vector.
            *pq-state* :parsing-required]
    (let [popp (fn [] (let [r (first *pq*)] (set! *pq* (rest *pq*)) r))
          peek #(first *pq*)]
      (loop [m {:required-names [], :required-args [], :specializers [] :optionals []  :keys []}]
        (let [result 
              (as-> m ?ll-map
                (cond (= '&optional (peek)) (do (popp) (set! *pq-state* :parsing-optional) ?ll-map), 
                      (= '&rest     (peek)) (do (popp) (set! *pq-state* :parsing-rest) ?ll-map),
                      (= '&key      (peek)) (do (popp) (set! *pq-state* :parsing-key) ?ll-map),
                      (= '&aok      (peek)) (do (popp) (assoc ?ll-map :allow-other-keys true)),
                      :else ?ll-map)
                (case *pq-state*
                  :parsing-required
                  (let [head (popp)
                        ?ll-map (update-in ?ll-map [:required-args] conj head)]
                    (if (list? head)
                      (update-in (update-in ?ll-map [:required-names] conj (first head))
                                 [:specializers] conj 
                                 (if (symbol? (second head)) 
                                   (qualified-symbol (second head)) ; a class
                                   (second head))) ; an eql specializer
                      (update-in (update-in ?ll-map [:required-names] conj head)
                                 [:specializers] conj true))),
                  :parsing-optional
                  (update-in ?ll-map [:optionals] conj (popp)),
                  :parsing-key
                  (update-in ?ll-map [:keys] conj (popp))))]
          (if (not-empty *pq*) (recur result) result))))))
                     
(defclass! std-generic-fn ()
  ((name :initarg :name)
   (lambda-list :initarg :lambda-list)
   (methods :initform [])
   (method-class :initarg :method-class)
   (descrim-fn :initarg :descrim-fn)
   (classes-to-emf-table :initform {})))

(defn ensure-generic-fn [fname lambda]
  (let [fname*  (if (resolve fname)
                  (symbol (str (.. (resolve fname) -ns -name))  (name fname))
                  (symbol (str (ns-name *ns*)) (name fname)))
        gf (make-instance 'cm/std-generic-fn :name fname* :lambda-list lambda)]
    (assoc-atom +gfuns+ fname* gf)
    gf))

(defn find-generic-fn
  [gname]
  (or (get @+gfuns+ gname)
      (throw (ex-info "No generic function:" {:name gname}))))

(defclass! std-method ()
  ((lambda-list :initarg :lambda-list)
   (qualifiers :initarg :qualifiers :initform [])
   (specializers :initarg :specializers :initform []) 
   (body :initarg :body)
   (generic-fn :initarg :generic-fn)
   (function :initform nil)))

(defn find-method 
  [gf qualifiers specializers & {:keys [error?] :or {error? true}}]
  (if-let [method (some #(when (and (= qualifiers   (getv % 'cm/qualifiers)) ; POD rewrite to return method?
                                    (= specializers (getv % 'cm/specializers)))
                           %)
                        (getv gf 'cm/methods))]
    method
    (when error?
      (throw 
       (ex-info "No such method" {:gf gf :qualifiers qualifiers :specializers specializers})))))

(defn- required-portion 
  [gf args]
  (let [number-required (count (:required-args (parse-lambda (getv gf 'cm/lambda-list))))] ; POD probably not!
    (when (< (count args) number-required)
      (throw (ex-info "Too few arguments to generic function:" {:gf gf :args args})))
    (take number-required args)))

;;; pg 284
(defn- sub-specializer? 
  "Returns true if C1 is a subclass of C2. A class is never a sub-specializer of itself."
  [c1 c2 c-arg]
  (some #(= % c2) (rest (drop-while #(not= c1 %) (getv (find-class c-arg) 'cm/cpl)))))

;;; Pg 303
(defn- std-method-more-specific?
  "Return -1 if m1 more specific than m2
           0 if m1 = m2
          +1 if m1 less specific than m2"
  [gf m1 m2 required-classes]
  (loop [spec1 (getv m1 'cm/specializers)
         spec2 (getv m2 'cm/specializers)
         arg-classes required-classes]
    (if (not= (first spec1) (first spec2))
      (if (sub-specializer? (first spec1) (first spec2) (first arg-classes))
        -1
        1)
      (if (empty? (rest spec1))
        0
        (recur (rest spec1) (rest spec2) (rest arg-classes))))))

(defn subclass? [c1 c2]
  "Returns true if c1 is a subclass of c2."
  (finalize-class c1)
  (finalize-class c2)
  (cond (= c2 true) true,
        (= c1 true) false,
        :else (some #(= % c2) (getv (find-class c1) 'cm/cpl))))

;;; POD I think required-portion already checks that method has same number of 
;;; specializers as required-classes. Check this. 
(defn- compute-applicable-methods-using-classes
  [gf required-classes]
  (sort ;<====== This is where order of execution is determined. 
   (fn [m1 m2] (std-method-more-specific? gf m1 m2 required-classes))
   (remove (fn [method] 
             (not
              (every? identity
                      (map (fn [r s] (subclass? r s))
                           required-classes
                           (getv method 'cm/specializers)))))
           (getv gf 'cm/methods))))

(defn- compute-primary-emfun
  [methods]
  (if (empty? methods) 
    nil
    (let [next-emfun (compute-primary-emfun (rest methods))]
      (fn [args] ((getv (first methods) 'cm/function) args next-emfun)))))

(defn primary-method? [method] (when (empty? (getv method 'cm/qualifiers)) method))
(defn before-method?  [method] (when (some #(when (= % :before) %) (getv method 'cm/qualifiers)) method))
(defn after-method?   [method] (when (some #(when (= % :after)  %) (getv method 'cm/qualifiers)) method))
(defn around-method?  [method] (when (some #(when (= % :around) %) (getv method 'cm/qualifiers)) method))

(defmacro ^:private prog1 [form & forms]
  "Return the result of the first form."
    `(let [f1# ~form]
       (do ~@forms)
       f1#))

;;; N.B. The parts of an effective method are called "effective method functions." Thus "emfun."
(defn- std-compute-effective-method-fn
  "Put together an effective method from the constitutent parts. Executed when a 
   method is executed, when the effective method has not yet been created (called from slow-method-lookup)."
  [gf methods]
  (let [primaries (remove (complement primary-method?) methods)
        around (some  #(around-method? %) methods)] ; Only wants most specific. Recursive. See below.
    (when (empty? primaries) (throw (ex-info "No suitable primary method." {:gf (getv gf 'cm/name)})))
    (if around
      (let [next-emfun (std-compute-effective-method-fn gf (remove #(= around %) methods))] 
        (fn [args] ((getv around 'cm/function) args next-emfun)))
      (let [next-emfun (compute-primary-emfun (rest primaries))
            befores (reverse (remove (complement before-method?) methods)) ; backwards from CLOS, always
            afters (remove (complement after-method?) methods)]            ; made more sense to me. 
        (fn [args] ; <===== The EMF !
          (doseq [b befores] ((getv b 'cm/function) args nil))
          (prog1 ; return value of primary, not some after.
           ((getv (first primaries) 'cm/function) args next-emfun)
           (doseq [a afters] ((getv a 'cm/function) args nil))))))))

(defn- slow-method-lookup
  [gf args classes]
  (let [applicable-methods (compute-applicable-methods-using-classes gf classes)
        emfun (std-compute-effective-method-fn gf applicable-methods)]
    (assocv gf 'cm/classes-to-emf-table 
            (assoc (getv gf 'cm/classes-to-emf-table) classes emfun))
    (emfun args)))

(defn- std-compute-descrim-fn
  [gf]
  (fn [ & args]
    (let [classes (map class-of (required-portion gf args))]
      (if-let [emfun (get (getv gf 'cm/classes-to-emf-table) classes)]
        (emfun args)
        (slow-method-lookup gf args classes)))))

(defn- finalize-generic-fn 
  [gf]
  (let [descrim-fn (std-compute-descrim-fn gf)]
    (assocv gf 'cm/descrim-fn descrim-fn)
    (if-let [var (resolve (getv gf 'cm/name))]
      (alter-var-root var (fn [& args] descrim-fn))
      (intern (symbol (namespace (getv gf 'cm/name)))
              (symbol (name (getv gf 'cm/name)))
              descrim-fn))
    (assocv gf 'cm/classes-to-emf-table {})
    true))

(defn- remove-method!
  [gf method]
  (assocv gf 'cm/methods (vec (remove #(= % method) (getv gf 'cm/methods))))
  (assocv method 'cm/generic-fn nil)
  (doseq [class (getv method 'cm/specializers)]
    (when-let [c (find-class class)]
      (assocv c
              'cm/direct-methods 
              (vec (remove #(= % method) (getv c 'cm/direct-methods))))))
  (finalize-generic-fn gf)
  method)

(defn- add-method 
  "Add the method to cm/direct-methods of the classes named as specializers."
  [gf method]
  (when-let [old-method (find-method gf 
                                     (getv method 'cm/qualifiers)
                                     (getv method 'cm/specializers)
                                     :error? false)]
    (remove-method! gf old-method))
  (assocv method 'cm/generic-fn gf)   
  (assocv gf 'cm/methods (conj (getv gf 'cm/methods) method))
  (doseq [specializer (getv method 'cm/specializers)] ;  :error? false
    (when (not= specializer true)
      (let [class (find-class specializer)] ; POD do I need a (obj-map method) here?
        (when-not (some #(= % (obj-map method)) (map obj-map (getv class 'cm/direct-methods))) ; :error? false
          (assocv class 'cm/direct-methods 
                  (conj (getv class 'cm/direct-methods) method))))))
  (finalize-generic-fn gf)
  method)

(defn- std-compute-method-fn
  "Create a method (part of effective method). Essentially this is something that wraps definitions of 
   call-next-method and next-method? and calls the body with arguments and the next effective method function. 
   Executed when defmethod! is evaluated."
  [method]
  (let [body (getv method 'cm/body),
        lambda-list (:required-names (getv method 'cm/lambda-list))]
    ;; POD Odd http://www.braveclojure.com/writing-macros/
    ;; "Syntax quoting will always include the symbol's full namespace" ..."to help you avoid name collisions" Thanks.
    (load-string  ; <================ Create a function --- Not pretty!
     (str 
      `(fn [~'args ~'next-emfun]
         (let [~'call-next-method (fn [& ~'cmn-args]
                                    (if (not ~'next-emfun)
                                      (throw (ex-info "No next method." 
                                                      {:method '~(getv (getv method 'cm/generic-fn) 'cm/name)}))
                                      (~'next-emfun (or ~'cmn-args ~'args)))),
               ~'next-method? (fn [] ~'next-emfun)]
           (apply (fn ~lambda-list ~@body) ~'args)))))))

(defn ensure-method [gf meth-map]
  (let [new-method
        (apply (partial make-instance 'cm/std-method :generic-fn gf)
               (mapcat (fn [[k v]] (list k v)) meth-map))]
    ;; In Closette, this is called from make-instance-standard-method.
    (assocv new-method 'cm/function (std-compute-method-fn new-method))
    (add-method gf new-method)
    new-method))

(defn parse-defmethod [args]
  (binding [*pq* args]
    (let [popp (fn [] (let [r (first *pq*)] (set! *pq* (rest *pq*)) r))]
      (as-> {:qualifiers []} ?arg-map
        (assoc ?arg-map :fname (popp))
        (if (keyword? (first *pq*)) ; POD as written can be only 1; okay?
          (update-in ?arg-map [:qualifiers] conj (popp))
          ?arg-map)
        (assoc ?arg-map :lambda-list (parse-lambda (popp)))
        (assoc ?arg-map :body *pq*)))))

(defmacro defgeneric! [fname lambda] ; POD & options NYI
  `(ensure-generic-fn '~fname '~lambda))

(defmacro defmethod! [& args]
  (let [arg-map# (parse-defmethod args)
        fname# (:fname arg-map#)]
    `(ensure-method (find-generic-fn
                     '~(if (resolve fname#)
                         (symbol (str (.. (resolve fname#) -ns -name))  (name fname#))
                         (symbol (str (ns-name *ns*)) (name fname#))))
                    {:lambda-list '~(:lambda-list arg-map#)
                     :qualifiers '~(:qualifiers arg-map#)
                     :specializers '~(:specializers (:lambda-list arg-map#))
                     :body '~(:body arg-map#)})))

(defgeneric! print-object [o w])

;;; POD Maybe this could just incorporate the type hint e.g. [(obj cm/std-object) (w ^java.io.Writer)]
(defmethod! print-object [(obj cm/std-object) w]
  (.write w (str "#<" (class-of obj) " " (obj-id obj)">")))

(defmethod! print-object [(obj cm/std-class) w]
  (.write w (str "#<" (class-of obj) " " (getv obj 'cm/name) ">")))

(defmethod! print-object [(obj cm/forward-ref-class) w]
  (.write w (str "#<{cm/forward-ref-class" " " (getv obj 'cm/name) ">")))

(defmethod print-method ClosmopObj [^gov.nist.closmop.ClosmopObj obj ^java.io.Writer w]
  (print-object obj w))

;;; It looks like it is seeing the first argument to p-r as a list, not a map.
(defmacro with-slots  
  "Similar to same-named CL special form, allow simplified expression of getting and setting.
   That is, do not make explicit reference to the object OBJ."
  [[& slots] obj & body]
  (let [o (gensym 'obj)] ; clj symbol# not getting it done.
    `(let [~o ~obj]
       ~@(postwalk-replace
          (zipmap slots (map (fn [x] `(getv ~o '~x false)) slots))  
          body))))

;;; Bootstrapping... (replacing the defn with a cm method)
(defgeneric! initialize-instance [instance initargs])

(defmethod! initialize-instance
  [instance initargs]
  (let [cname (class-of instance)
        iforms (apply hash-map
                      (mapcat (fn [[sname m]] 
                                (when-let [form (:initform m)]
                                  (list (first (:initargs m)) (eval form))))
                              (getv (find-class cname) 'cm/effective-slots)))]
    (doseq [[k v] (merge iforms initargs)]
      (assocv instance (first (find-slot-by-initarg cname k)) v)))
  instance)
;;; ...End bootstrapping.


