/*
 * Decompiled with CFR 0.152.
 */
package com.thinkaurelius.titan.graphdb.database.idassigner;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.thinkaurelius.titan.core.TitanException;
import com.thinkaurelius.titan.diskstorage.BackendException;
import com.thinkaurelius.titan.diskstorage.IDAuthority;
import com.thinkaurelius.titan.diskstorage.IDBlock;
import com.thinkaurelius.titan.graphdb.database.idassigner.IDPool;
import com.thinkaurelius.titan.graphdb.database.idassigner.IDPoolExhaustedException;
import java.time.Duration;
import java.util.ArrayDeque;
import java.util.Queue;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StandardIDPool
implements IDPool {
    private static final Logger log = LoggerFactory.getLogger(StandardIDPool.class);
    private static final IDBlock ID_POOL_EXHAUSTION = new IDBlock(){

        @Override
        public long numIds() {
            throw new UnsupportedOperationException();
        }

        @Override
        public long getId(long index) {
            throw new UnsupportedOperationException();
        }
    };
    private static final IDBlock UNINITIALIZED_BLOCK = new IDBlock(){

        @Override
        public long numIds() {
            return 0L;
        }

        @Override
        public long getId(long index) {
            throw new ArrayIndexOutOfBoundsException(0);
        }
    };
    private static final int RENEW_ID_COUNT = 100;
    private final IDAuthority idAuthority;
    private final long idUpperBound;
    private final int partition;
    private final int idNamespace;
    private final Duration renewTimeout;
    private final double renewBufferPercentage;
    private IDBlock currentBlock;
    private long currentIndex;
    private long renewBlockIndex;
    private volatile IDBlock nextBlock;
    private Future<IDBlock> idBlockFuture;
    private IDBlockGetter idBlockGetter;
    private final ThreadPoolExecutor exec;
    private volatile boolean closed;
    private final Queue<Future<?>> closeBlockers;

    public StandardIDPool(IDAuthority idAuthority, int partition, int idNamespace, long idUpperBound, Duration renewTimeout, double renewBufferPercentage) {
        Preconditions.checkArgument((idUpperBound > 0L ? 1 : 0) != 0);
        this.idAuthority = idAuthority;
        Preconditions.checkArgument((partition >= 0 ? 1 : 0) != 0);
        this.partition = partition;
        Preconditions.checkArgument((idNamespace >= 0 ? 1 : 0) != 0);
        this.idNamespace = idNamespace;
        this.idUpperBound = idUpperBound;
        Preconditions.checkArgument((!renewTimeout.isZero() ? 1 : 0) != 0, (Object)"Renew-timeout must be positive");
        this.renewTimeout = renewTimeout;
        Preconditions.checkArgument((renewBufferPercentage > 0.0 && renewBufferPercentage <= 1.0 ? 1 : 0) != 0, (Object)"Renew-buffer percentage must be in (0.0,1.0]");
        this.renewBufferPercentage = renewBufferPercentage;
        this.currentBlock = UNINITIALIZED_BLOCK;
        this.currentIndex = 0L;
        this.renewBlockIndex = 0L;
        this.nextBlock = null;
        this.exec = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(), new ThreadFactoryBuilder().setDaemon(false).setNameFormat("TitanID(" + partition + ")(" + idNamespace + ")[%d]").build());
        this.idBlockFuture = null;
        this.closeBlockers = new ArrayDeque(4);
        this.closed = false;
    }

    private synchronized void waitForIDBlockGetter() throws InterruptedException {
        Stopwatch sw = Stopwatch.createStarted();
        if (null != this.idBlockFuture) {
            try {
                this.nextBlock = this.idBlockFuture.get(this.renewTimeout.toMillis(), TimeUnit.MILLISECONDS);
            }
            catch (ExecutionException e) {
                String msg = String.format("ID block allocation on partition(%d)-namespace(%d) failed with an exception in %s", this.partition, this.idNamespace, sw.stop());
                throw new TitanException(msg, e);
            }
            catch (TimeoutException e) {
                String msg = String.format("ID block allocation on partition(%d)-namespace(%d) timed out in %s", this.partition, this.idNamespace, sw.stop());
                this.idBlockGetter.stopRequested();
                if (this.idAuthority.supportsInterruption()) {
                    this.idBlockFuture.cancel(true);
                } else {
                    Future<?> f;
                    if (!this.closeBlockers.isEmpty() && null != (f = this.closeBlockers.peek()) && f.isDone()) {
                        this.closeBlockers.remove();
                    }
                    this.closeBlockers.add(this.idBlockFuture);
                }
                throw new TitanException(msg, e);
            }
            catch (CancellationException e) {
                String msg = String.format("ID block allocation on partition(%d)-namespace(%d) was cancelled after %s", this.partition, this.idNamespace, sw.stop());
                throw new TitanException(msg, e);
            }
            finally {
                this.idBlockFuture = null;
            }
        }
    }

    private synchronized void nextBlock() throws InterruptedException {
        assert (this.currentIndex == this.currentBlock.numIds());
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (String)"ID Pool has been closed for partition(%s)-namespace(%s) - cannot apply for new id block", (Object[])new Object[]{this.partition, this.idNamespace});
        if (null == this.nextBlock && null == this.idBlockFuture) {
            this.startIDBlockGetter();
        }
        if (null == this.nextBlock) {
            this.waitForIDBlockGetter();
        }
        if (this.nextBlock == ID_POOL_EXHAUSTION) {
            throw new IDPoolExhaustedException("Exhausted ID Pool for partition(" + this.partition + ")-namespace(" + this.idNamespace + ")");
        }
        this.currentBlock = this.nextBlock;
        this.currentIndex = 0L;
        log.debug("ID partition({})-namespace({}) acquired block: [{}]", new Object[]{this.partition, this.idNamespace, this.currentBlock});
        assert (this.currentBlock.numIds() > 0L);
        this.nextBlock = null;
        this.renewBlockIndex = Math.max(0L, this.currentBlock.numIds() - Math.max(100L, Math.round((double)this.currentBlock.numIds() * this.renewBufferPercentage)));
        assert (this.renewBlockIndex < this.currentBlock.numIds() && this.renewBlockIndex >= this.currentIndex);
    }

    @Override
    public synchronized long nextID() {
        assert (this.currentIndex <= this.currentBlock.numIds());
        if (this.currentIndex == this.currentBlock.numIds()) {
            try {
                this.nextBlock();
            }
            catch (InterruptedException e) {
                throw new TitanException("Could not renew id block due to interruption", e);
            }
        }
        if (this.currentIndex == this.renewBlockIndex) {
            this.startIDBlockGetter();
        }
        long returnId = this.currentBlock.getId(this.currentIndex);
        ++this.currentIndex;
        if (returnId >= this.idUpperBound) {
            throw new IDPoolExhaustedException("Reached id upper bound of " + this.idUpperBound);
        }
        log.trace("partition({})-namespace({}) Returned id: {}", new Object[]{this.partition, this.idNamespace, returnId});
        return returnId;
    }

    @Override
    public synchronized void close() {
        this.closed = true;
        try {
            this.waitForIDBlockGetter();
        }
        catch (InterruptedException e) {
            throw new TitanException("Interrupted while waiting for id renewer thread to finish", e);
        }
        for (Future future : this.closeBlockers) {
            try {
                future.get();
            }
            catch (InterruptedException e) {
                throw new TitanException("Interrupted while waiting for runaway ID renewer task " + future, e);
            }
            catch (ExecutionException e) {
                log.debug("Runaway ID renewer task completed with exception", (Throwable)e);
            }
        }
        this.exec.shutdownNow();
    }

    private synchronized void startIDBlockGetter() {
        Preconditions.checkArgument((this.idBlockFuture == null ? 1 : 0) != 0, this.idBlockFuture);
        if (this.closed) {
            return;
        }
        log.debug("Starting id block renewal thread upon {}", (Object)this.currentIndex);
        this.idBlockGetter = new IDBlockGetter(this.idAuthority, this.partition, this.idNamespace, this.renewTimeout);
        this.idBlockFuture = this.exec.submit(this.idBlockGetter);
    }

    private static class IDBlockGetter
    implements Callable<IDBlock> {
        private final Stopwatch alive;
        private final IDAuthority idAuthority;
        private final int partition;
        private final int idNamespace;
        private final Duration renewTimeout;
        private volatile boolean stopRequested;

        public IDBlockGetter(IDAuthority idAuthority, int partition, int idNamespace, Duration renewTimeout) {
            this.idAuthority = idAuthority;
            this.partition = partition;
            this.idNamespace = idNamespace;
            this.renewTimeout = renewTimeout;
            this.alive = Stopwatch.createStarted();
        }

        private void stopRequested() {
            this.stopRequested = true;
        }

        @Override
        public IDBlock call() {
            Stopwatch running = Stopwatch.createStarted();
            try {
                if (this.stopRequested) {
                    log.debug("Aborting ID block retrieval on partition({})-namespace({}) after graceful shutdown was requested, exec time {}, exec+q time {}", new Object[]{this.partition, this.idNamespace, running.stop(), this.alive.stop()});
                    throw new TitanException("ID block retrieval aborted by caller");
                }
                IDBlock idBlock = this.idAuthority.getIDBlock(this.partition, this.idNamespace, this.renewTimeout);
                log.debug("Retrieved ID block from authority on partition({})-namespace({}), exec time {}, exec+q time {}", new Object[]{this.partition, this.idNamespace, running.stop(), this.alive.stop()});
                Preconditions.checkArgument((idBlock != null && idBlock.numIds() > 0L ? 1 : 0) != 0);
                return idBlock;
            }
            catch (BackendException e) {
                throw new TitanException("Could not acquire new ID block from storage", e);
            }
            catch (IDPoolExhaustedException e) {
                return ID_POOL_EXHAUSTION;
            }
        }
    }
}

