/*
 * Decompiled with CFR 0.152.
 */
package com.thinkaurelius.titan.core.attribute;

import com.google.common.base.Preconditions;
import com.google.common.primitives.Doubles;
import com.spatial4j.core.context.SpatialContextFactory;
import com.spatial4j.core.context.jts.DatelineRule;
import com.spatial4j.core.context.jts.JtsSpatialContext;
import com.spatial4j.core.context.jts.JtsSpatialContextFactory;
import com.spatial4j.core.distance.DistanceUtils;
import com.spatial4j.core.io.jts.JtsBinaryCodec;
import com.spatial4j.core.io.jts.JtsGeoJSONReader;
import com.spatial4j.core.io.jts.JtsGeoJSONWriter;
import com.spatial4j.core.io.jts.JtsWKTReader;
import com.spatial4j.core.io.jts.JtsWKTWriter;
import com.spatial4j.core.shape.Circle;
import com.spatial4j.core.shape.Rectangle;
import com.spatial4j.core.shape.Shape;
import com.spatial4j.core.shape.SpatialRelation;
import com.spatial4j.core.shape.jts.JtsGeometry;
import com.thinkaurelius.titan.core.attribute.AttributeSerializer;
import com.thinkaurelius.titan.diskstorage.ScanBuffer;
import com.thinkaurelius.titan.diskstorage.WriteBuffer;
import com.thinkaurelius.titan.graphdb.database.idhandling.VariableLong;
import com.vividsolutions.jts.geom.Coordinate;
import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.GeometryFactory;
import com.vividsolutions.jts.geom.Polygon;
import java.io.ByteArrayInputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.lang.reflect.Array;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.io.output.ByteArrayOutputStream;
import org.apache.tinkerpop.gremlin.structure.io.graphson.GraphSONUtil;
import org.apache.tinkerpop.shaded.jackson.core.JsonGenerator;
import org.apache.tinkerpop.shaded.jackson.core.JsonParser;
import org.apache.tinkerpop.shaded.jackson.core.JsonProcessingException;
import org.apache.tinkerpop.shaded.jackson.databind.DeserializationContext;
import org.apache.tinkerpop.shaded.jackson.databind.ObjectMapper;
import org.apache.tinkerpop.shaded.jackson.databind.SerializerProvider;
import org.apache.tinkerpop.shaded.jackson.databind.deser.std.StdDeserializer;
import org.apache.tinkerpop.shaded.jackson.databind.jsontype.TypeSerializer;
import org.apache.tinkerpop.shaded.jackson.databind.ser.std.StdSerializer;
import org.apache.tinkerpop.shaded.kryo.Kryo;
import org.apache.tinkerpop.shaded.kryo.Serializer;
import org.apache.tinkerpop.shaded.kryo.io.Input;
import org.apache.tinkerpop.shaded.kryo.io.Output;

public class Geoshape {
    private static String FIELD_LABEL = "geometry";
    private static String FIELD_TYPE = "type";
    private static String FIELD_COORDINATES = "coordinates";
    private static String FIELD_RADIUS = "radius";
    public static final JtsSpatialContext CTX;
    private static JtsWKTReader wktReader;
    private static JtsWKTWriter wktWriter;
    private static JtsGeoJSONReader geojsonReader;
    private static JtsGeoJSONWriter geojsonWriter;
    private final Shape shape;

    private Geoshape(Shape shape) {
        Preconditions.checkNotNull((Object)shape, (Object)"Invalid shape (null)");
        this.shape = shape;
    }

    public int hashCode() {
        return this.shape.hashCode();
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other == null) {
            return false;
        }
        if (!this.getClass().isInstance(other)) {
            return false;
        }
        Geoshape oth = (Geoshape)other;
        return this.shape.equals((Object)oth.shape);
    }

    public String toString() {
        return wktWriter.toString(this.shape);
    }

    public String toGeoJson() {
        return GeoshapeGsonSerializer.toGeoJson(this);
    }

    public Shape getShape() {
        return this.shape;
    }

    public Type getType() {
        Type type;
        if (com.spatial4j.core.shape.Point.class.isAssignableFrom(this.shape.getClass())) {
            type = Type.POINT;
        } else if (Circle.class.isAssignableFrom(this.shape.getClass())) {
            type = Type.CIRCLE;
        } else if (Rectangle.class.isAssignableFrom(this.shape.getClass())) {
            type = Type.BOX;
        } else if (JtsGeometry.class.isAssignableFrom(this.shape.getClass()) && "LineString".equals(((JtsGeometry)this.shape).getGeom().getGeometryType())) {
            type = Type.LINE;
        } else if (JtsGeometry.class.isAssignableFrom(this.shape.getClass())) {
            try {
                type = Type.fromGson(((JtsGeometry)this.shape).getGeom().getGeometryType());
            }
            catch (IllegalArgumentException e) {
                throw new IllegalStateException("Unrecognized shape type");
            }
        } else {
            throw new IllegalStateException("Unrecognized shape type");
        }
        return type;
    }

    public int size() {
        switch (this.getType()) {
            case POINT: {
                return 1;
            }
            case CIRCLE: {
                return 1;
            }
            case BOX: {
                return 2;
            }
            case LINE: 
            case POLYGON: {
                return ((JtsGeometry)this.shape).getGeom().getCoordinates().length;
            }
        }
        throw new IllegalStateException("size() not supported for type: " + (Object)((Object)this.getType()));
    }

    public Point getPoint(int position) {
        if (position < 0 || position >= this.size()) {
            throw new ArrayIndexOutOfBoundsException("Invalid position: " + position);
        }
        switch (this.getType()) {
            case POINT: 
            case CIRCLE: {
                return this.getPoint();
            }
            case BOX: {
                if (position == 0) {
                    return new Point(this.shape.getBoundingBox().getMinY(), this.shape.getBoundingBox().getMinX());
                }
                return new Point(this.shape.getBoundingBox().getMaxY(), this.shape.getBoundingBox().getMaxX());
            }
            case LINE: 
            case POLYGON: {
                Coordinate coordinate = ((JtsGeometry)this.shape).getGeom().getCoordinates()[position];
                return new Point(coordinate.y, coordinate.x);
            }
        }
        throw new IllegalStateException("getPoint(int) not supported for type: " + (Object)((Object)this.getType()));
    }

    public Point getPoint() {
        Preconditions.checkArgument((this.getType() == Type.POINT || this.getType() == Type.CIRCLE ? 1 : 0) != 0, (Object)"Shape does not have a single point");
        return new Point(this.shape.getCenter().getY(), this.shape.getCenter().getX());
    }

    public double getRadius() {
        Preconditions.checkArgument((this.getType() == Type.CIRCLE ? 1 : 0) != 0, (Object)"This shape is not a circle");
        double radiusInDeg = ((Circle)this.shape).getRadius();
        return DistanceUtils.degrees2Dist((double)radiusInDeg, (double)6371.0087714);
    }

    private SpatialRelation getSpatialRelation(Geoshape other) {
        Preconditions.checkNotNull((Object)other);
        return this.shape.relate(other.shape);
    }

    public boolean intersect(Geoshape other) {
        SpatialRelation r = this.getSpatialRelation(other);
        return r == SpatialRelation.INTERSECTS || r == SpatialRelation.CONTAINS || r == SpatialRelation.WITHIN;
    }

    public boolean within(Geoshape outer) {
        return this.getSpatialRelation(outer) == SpatialRelation.WITHIN;
    }

    public boolean contains(Geoshape outer) {
        return this.getSpatialRelation(outer) == SpatialRelation.CONTAINS;
    }

    public boolean disjoint(Geoshape other) {
        return this.getSpatialRelation(other) == SpatialRelation.DISJOINT;
    }

    public static final Geoshape point(double latitude, double longitude) {
        Preconditions.checkArgument((boolean)Geoshape.isValidCoordinate(latitude, longitude), (Object)"Invalid coordinate provided");
        return new Geoshape((Shape)CTX.makePoint(longitude, latitude));
    }

    public static final Geoshape circle(double latitude, double longitude, double radiusInKM) {
        Preconditions.checkArgument((boolean)Geoshape.isValidCoordinate(latitude, longitude), (Object)"Invalid coordinate provided");
        Preconditions.checkArgument((radiusInKM > 0.0 ? 1 : 0) != 0, (String)"Invalid radius provided [%s]", (Object[])new Object[]{radiusInKM});
        double radius = DistanceUtils.dist2Degrees((double)radiusInKM, (double)6371.0087714);
        return new Geoshape((Shape)CTX.makeCircle(longitude, latitude, radius));
    }

    public static final Geoshape box(double southWestLatitude, double southWestLongitude, double northEastLatitude, double northEastLongitude) {
        Preconditions.checkArgument((boolean)Geoshape.isValidCoordinate(southWestLatitude, southWestLongitude), (Object)"Invalid south-west coordinate provided");
        Preconditions.checkArgument((boolean)Geoshape.isValidCoordinate(northEastLatitude, northEastLongitude), (Object)"Invalid north-east coordinate provided");
        return new Geoshape((Shape)CTX.makeRectangle(southWestLongitude, northEastLongitude, southWestLatitude, northEastLatitude));
    }

    public static final Geoshape line(double ... coordinates) {
        Preconditions.checkArgument((coordinates.length % 2 == 0 ? 1 : 0) != 0, (Object)"Odd number of coordinates provided");
        Preconditions.checkArgument((coordinates.length >= 4 ? 1 : 0) != 0, (Object)"Too few coordinate pairs provided");
        ArrayList<double[]> points = new ArrayList<double[]>();
        for (int i = 0; i < coordinates.length; i += 2) {
            points.add(new double[]{coordinates[i + 1], coordinates[i]});
        }
        return Geoshape.line(points);
    }

    public static final Geoshape line(List<double[]> coordinates) {
        Preconditions.checkArgument((coordinates.size() >= 2 ? 1 : 0) != 0, (Object)"Too few coordinate pairs provided");
        ArrayList<com.spatial4j.core.shape.Point> points = new ArrayList<com.spatial4j.core.shape.Point>();
        for (double[] coordinate : coordinates) {
            Preconditions.checkArgument((boolean)Geoshape.isValidCoordinate(coordinate[1], coordinate[0]), (Object)"Invalid coordinate provided");
            points.add(CTX.makePoint(coordinate[0], coordinate[1]));
        }
        return new Geoshape(CTX.makeLineString(points));
    }

    public static final Geoshape polygon(double ... coordinates) {
        Preconditions.checkArgument((coordinates.length % 2 == 0 ? 1 : 0) != 0, (Object)"Odd number of coordinates provided");
        Preconditions.checkArgument((coordinates.length >= 6 ? 1 : 0) != 0, (Object)"Too few coordinate pairs provided");
        ArrayList<double[]> points = new ArrayList<double[]>();
        for (int i = 0; i < coordinates.length; i += 2) {
            points.add(new double[]{coordinates[i + 1], coordinates[i]});
        }
        return Geoshape.polygon(points);
    }

    public static final Geoshape polygon(List<double[]> coordinates) {
        Preconditions.checkArgument((coordinates.size() >= 3 ? 1 : 0) != 0, (Object)"Too few coordinate pairs provided");
        Coordinate[] points = new Coordinate[coordinates.size()];
        for (int i = 0; i < coordinates.size(); ++i) {
            Preconditions.checkArgument((coordinates.get(i).length >= 2 ? 1 : 0) != 0, (Object)"Too few coordinates in pair");
            Preconditions.checkArgument((boolean)Geoshape.isValidCoordinate(coordinates.get(i)[1], coordinates.get(i)[0]), (Object)"Invalid coordinate provided");
            points[i] = new Coordinate(coordinates.get(i)[0], coordinates.get(i)[1]);
        }
        Polygon polygon = new GeometryFactory().createPolygon(points);
        return new Geoshape((Shape)CTX.makeShape((Geometry)polygon));
    }

    public static final Geoshape geoshape(Geometry geometry) {
        return new Geoshape((Shape)CTX.makeShape(geometry));
    }

    public static final Geoshape geoshape(Shape shape) {
        return new Geoshape(shape);
    }

    public static final Geoshape fromWkt(String wkt) throws ParseException {
        return new Geoshape(wktReader.parse(wkt));
    }

    public static final boolean isValidCoordinate(double latitude, double longitude) {
        return latitude >= -90.0 && latitude <= 90.0 && longitude >= -180.0 && longitude <= 180.0;
    }

    static {
        JtsSpatialContextFactory factory = new JtsSpatialContextFactory();
        factory.geo = true;
        factory.useJtsPoint = false;
        factory.useJtsLineString = true;
        factory.datelineRule = DatelineRule.none;
        CTX = new JtsSpatialContext(factory);
        wktReader = new JtsWKTReader(CTX, new JtsSpatialContextFactory());
        wktWriter = new JtsWKTWriter(CTX, new JtsSpatialContextFactory());
        geojsonReader = new JtsGeoJSONReader(CTX, new SpatialContextFactory());
        geojsonWriter = new JtsGeoJSONWriter(CTX, new SpatialContextFactory());
    }

    public static class GeoshapeBinarySerializer {
        private static JtsBinaryCodec binaryCodec = new JtsBinaryCodec(CTX, new JtsSpatialContextFactory());

        public static void write(OutputStream outputStream, Geoshape attribute) throws IOException {
            outputStream.write(attribute.shape instanceof JtsGeometry ? 0 : 1);
            try (DataOutputStream dataOutput = new DataOutputStream(outputStream);){
                if (attribute.shape instanceof JtsGeometry) {
                    binaryCodec.writeJtsGeom((DataOutput)dataOutput, attribute.shape);
                } else {
                    binaryCodec.writeShape((DataOutput)dataOutput, attribute.shape);
                }
                dataOutput.flush();
            }
            outputStream.flush();
        }

        public static Geoshape read(InputStream inputStream) throws IOException {
            boolean isJts = inputStream.read() == 0;
            try (DataInputStream dataInput = new DataInputStream(inputStream);){
                if (isJts) {
                    Geoshape geoshape = new Geoshape(binaryCodec.readJtsGeom((DataInput)dataInput));
                    return geoshape;
                }
                Geoshape geoshape = new Geoshape(binaryCodec.readShape((DataInput)dataInput));
                return geoshape;
            }
        }
    }

    public static class GeoshapeGsonDeserializer
    extends StdDeserializer<Geoshape> {
        public GeoshapeGsonDeserializer() {
            super(Geoshape.class);
        }

        public Geoshape deserialize(JsonParser jsonParser, DeserializationContext deserializationContext) throws IOException, JsonProcessingException {
            jsonParser.nextToken();
            if (jsonParser.getCurrentName().equals("coordinates")) {
                double[] f = (double[])jsonParser.readValueAs(double[].class);
                jsonParser.nextToken();
                return Geoshape.point(f[1], f[0]);
            }
            try {
                HashMap map = (HashMap)jsonParser.readValueAs(LinkedHashMap.class);
                jsonParser.nextToken();
                String json = new ObjectMapper().writeValueAsString((Object)map);
                Geoshape shape = new Geoshape(geojsonReader.read((Reader)new StringReader(json)));
                return shape;
            }
            catch (ParseException e) {
                throw new IOException("Unable to read and parse geojson", e);
            }
        }
    }

    public static class GeoshapeGsonSerializer
    extends StdSerializer<Geoshape> {
        public GeoshapeGsonSerializer() {
            super(Geoshape.class);
        }

        public void serialize(Geoshape value, JsonGenerator jgen, SerializerProvider provider) throws IOException, JsonProcessingException {
            switch (value.getType()) {
                case POINT: {
                    jgen.writeStartObject();
                    jgen.writeFieldName(FIELD_TYPE);
                    jgen.writeString(Type.POINT.toString());
                    jgen.writeFieldName(FIELD_COORDINATES);
                    jgen.writeStartArray();
                    jgen.writeNumber(value.getPoint().getLongitude());
                    jgen.writeNumber(value.getPoint().getLatitude());
                    jgen.writeEndArray();
                    jgen.writeEndObject();
                    break;
                }
                default: {
                    jgen.writeRawValue(GeoshapeGsonSerializer.toGeoJson(value));
                }
            }
        }

        public void serializeWithType(Geoshape geoshape, JsonGenerator jgen, SerializerProvider serializerProvider, TypeSerializer typeSerializer) throws IOException, JsonProcessingException {
            jgen.writeStartObject();
            if (typeSerializer != null) {
                jgen.writeStringField("@class", Geoshape.class.getName());
            }
            String geojson = GeoshapeGsonSerializer.toGeoJson(geoshape);
            Map json = (Map)new ObjectMapper().readValue(geojson, LinkedHashMap.class);
            if (geoshape.getType() == Type.POINT) {
                double[] coords = ((List)json.get("coordinates")).stream().map(i -> i.doubleValue()).mapToDouble(i -> i).toArray();
                GraphSONUtil.writeWithType((String)FIELD_COORDINATES, (Object)coords, (JsonGenerator)jgen, (SerializerProvider)serializerProvider, (TypeSerializer)typeSerializer);
            } else {
                GraphSONUtil.writeWithType((String)FIELD_LABEL, (Object)json, (JsonGenerator)jgen, (SerializerProvider)serializerProvider, (TypeSerializer)typeSerializer);
            }
            jgen.writeEndObject();
        }

        public static String toGeoJson(Geoshape geoshape) {
            return geojsonWriter.toString(geoshape.shape);
        }
    }

    public static class GeoShapeGryoSerializer
    extends Serializer<Geoshape> {
        public void write(Kryo kryo, Output output, Geoshape geoshape) {
            try {
                ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
                GeoshapeBinarySerializer.write((OutputStream)outputStream, geoshape);
                byte[] bytes = outputStream.toByteArray();
                output.write(bytes.length);
                output.write(bytes);
            }
            catch (IOException e) {
                throw new RuntimeException("I/O exception writing geoshape");
            }
        }

        public Geoshape read(Kryo kryo, Input input, Class<Geoshape> aClass) {
            int length = input.read();
            assert (length > 0);
            ByteArrayInputStream inputStream = new ByteArrayInputStream(input.readBytes(length));
            try {
                return GeoshapeBinarySerializer.read(inputStream);
            }
            catch (IOException e) {
                throw new RuntimeException("I/O exception reding geoshape");
            }
        }
    }

    public static class GeoshapeSerializer
    implements AttributeSerializer<Geoshape> {
        @Override
        public void verifyAttribute(Geoshape value) {
        }

        @Override
        public Geoshape convert(Object value) {
            if (value instanceof Map) {
                return this.convertGeoJson(value);
            }
            if (value instanceof Collection) {
                value = this.convertCollection((Collection)value);
            }
            if (value.getClass().isArray() && (value.getClass().getComponentType().isPrimitive() || Number.class.isAssignableFrom(value.getClass().getComponentType()))) {
                Geoshape shape = null;
                int len = Array.getLength(value);
                double[] arr = new double[len];
                for (int i = 0; i < len; ++i) {
                    arr[i] = ((Number)Array.get(value, i)).doubleValue();
                }
                if (len == 2) {
                    shape = Geoshape.point(arr[0], arr[1]);
                } else if (len == 3) {
                    shape = Geoshape.circle(arr[0], arr[1], arr[2]);
                } else if (len == 4) {
                    shape = Geoshape.box(arr[0], arr[1], arr[2], arr[3]);
                } else {
                    throw new IllegalArgumentException("Expected 2-4 coordinates to create Geoshape, but given: " + value);
                }
                return shape;
            }
            if (value instanceof String) {
                String delimiter;
                String[] components = null;
                String[] len = new String[]{",", ";"};
                int arr = len.length;
                for (int i = 0; i < arr && ((components = ((String)value).split(delimiter = len[i])).length < 2 || components.length > 4); ++i) {
                    components = null;
                }
                Preconditions.checkArgument((components != null ? 1 : 0) != 0, (String)"Could not parse coordinates from string: %s", (Object[])new Object[]{value});
                double[] coords = new double[components.length];
                try {
                    for (int i = 0; i < components.length; ++i) {
                        coords[i] = Double.parseDouble(components[i]);
                    }
                }
                catch (NumberFormatException e) {
                    throw new IllegalArgumentException("Could not parse coordinates from string: " + value, e);
                }
                return this.convert(coords);
            }
            return null;
        }

        private double[] convertCollection(Collection<Object> c) {
            List numbers = c.stream().map(o -> {
                if (!(o instanceof Number)) {
                    throw new IllegalArgumentException("Collections may only contain numbers to create a Geoshape");
                }
                return ((Number)o).doubleValue();
            }).collect(Collectors.toList());
            return Doubles.toArray(numbers);
        }

        private Geoshape convertGeoJson(Object value) {
            try {
                Map map = (Map)value;
                String type = (String)map.get("type");
                if ("Feature".equals(type)) {
                    Map geometry = (Map)map.get("geometry");
                    return this.convertGeometry(geometry);
                }
                return this.convertGeometry(map);
            }
            catch (IOException | ClassCastException | ParseException e) {
                throw new IllegalArgumentException("GeoJSON was unparsable");
            }
        }

        private Geoshape convertGeometry(Map<String, Object> geometry) throws IOException, ParseException {
            String type = (String)geometry.get("type");
            List coordinates = (List)geometry.get("coordinates");
            if ("Point".equals(type)) {
                double[] parsedCoordinates = this.convertCollection(coordinates);
                return Geoshape.point(parsedCoordinates[1], parsedCoordinates[0]);
            }
            if ("Circle".equals(type)) {
                Number radius = (Number)geometry.get("radius");
                if (radius == null) {
                    throw new IllegalArgumentException("GeoJSON circles require a radius");
                }
                double[] parsedCoordinates = this.convertCollection(coordinates);
                return Geoshape.circle(parsedCoordinates[1], parsedCoordinates[0], radius.doubleValue());
            }
            if ("Polygon".equals(type) && coordinates.size() == 4) {
                double[] p0 = this.convertCollection((Collection)coordinates.get(0));
                double[] p1 = this.convertCollection((Collection)coordinates.get(1));
                double[] p2 = this.convertCollection((Collection)coordinates.get(2));
                double[] p3 = this.convertCollection((Collection)coordinates.get(3));
                if (p0[0] == p1[0] && p1[1] == p2[1] && p2[0] == p3[0] && p3[1] == p0[1] && p3[0] != p0[0] || p0[1] == p1[1] && p1[0] == p2[0] && p2[1] == p3[1] && p3[0] == p0[0] && p3[1] != p0[1]) {
                    return Geoshape.box(this.min(p0[1], p1[1], p2[1], p3[1]), this.min(p0[0], p1[0], p2[0], p3[0]), this.max(p0[1], p1[1], p2[1], p3[1]), this.max(p0[0], p1[0], p2[0], p3[0]));
                }
            }
            String json = new ObjectMapper().writeValueAsString(geometry);
            return new Geoshape(geojsonReader.read((Reader)new StringReader(json)));
        }

        private double min(double ... numbers) {
            return Arrays.stream(numbers).min().getAsDouble();
        }

        private double max(double ... numbers) {
            return Arrays.stream(numbers).max().getAsDouble();
        }

        @Override
        public Geoshape read(ScanBuffer buffer) {
            long l = VariableLong.readPositive(buffer);
            assert (l > 0L && l < Integer.MAX_VALUE);
            int length = (int)l;
            ByteArrayInputStream inputStream = new ByteArrayInputStream(buffer.getBytes(length));
            try {
                return GeoshapeBinarySerializer.read(inputStream);
            }
            catch (IOException e) {
                throw new RuntimeException("I/O exception reading geoshape");
            }
        }

        @Override
        public void write(WriteBuffer buffer, Geoshape attribute) {
            try {
                ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
                GeoshapeBinarySerializer.write((OutputStream)outputStream, attribute);
                byte[] bytes = outputStream.toByteArray();
                VariableLong.writePositive(buffer, bytes.length);
                buffer.putBytes(bytes);
            }
            catch (IOException e) {
                throw new RuntimeException("I/O exception writing geoshape");
            }
        }
    }

    public static final class Point {
        private final double longitude;
        private final double latitude;

        Point(double latitude, double longitude) {
            this.longitude = longitude;
            this.latitude = latitude;
        }

        public double getLongitude() {
            return this.longitude;
        }

        public double getLatitude() {
            return this.latitude;
        }

        private com.spatial4j.core.shape.Point getSpatial4jPoint() {
            return CTX.makePoint(this.longitude, this.latitude);
        }

        public double distance(Point other) {
            return DistanceUtils.degrees2Dist((double)CTX.getDistCalc().distance(this.getSpatial4jPoint(), other.getSpatial4jPoint()), (double)6371.0087714);
        }
    }

    public static enum Type {
        POINT("Point"),
        BOX("Box"),
        CIRCLE("Circle"),
        LINE("Line"),
        POLYGON("Polygon"),
        MULTIPOINT("MultiPoint"),
        MULTILINESTRING("MultiLineString"),
        MULTIPOLYGON("MultiPolygon");

        private final String gsonName;

        private Type(String gsonName) {
            this.gsonName = gsonName;
        }

        public boolean gsonEquals(String otherGson) {
            return otherGson != null && this.gsonName.equals(otherGson);
        }

        public static Type fromGson(String gsonShape) {
            return Type.valueOf(gsonShape.toUpperCase());
        }

        public String toString() {
            return this.gsonName;
        }
    }
}

