(ns gorillalabs.config
  (:import [java.io PushbackReader]
           [java.net URL MalformedURLException])
  (:refer-clojure :exclude [read])
  (:require [clojure.java.io :as io]
            [clojure.edn :refer [read]]))

;; The base version is brought to you by Craig Andera (see https://gist.github.com/candera/4565367),
;; we just improved on it.

;;; A little Clojure configuration reader
;;; =====================================
;;;
;;; This is a handy bit of code I've written more than once.
;;; I thought I'd throw it in here so I can refer back to it later.
;;; Basically, it lets you read a config file to produce a Clojure map.
;;; The config files themselves can contain one or more forms, each of which can be either a map or a list.
;;; Maps are simply merged.
;;; Lists correspond to invocations of extension points, which in turn produces a map, which is merged along with everything else.


;; Gorillalabs Changelog:
;; - Switched to clojure.edn to prevent execution of code from config files
;; - fixed problem with in-JAR-URLs (like jar:file:///home/user/test.jar!config.edn)
;; - allowing config for different environments (config@ENVIRONMENT.edn)



(defn form-seq
  "Lazy seq of forms read from a reader"
  [reader]
  (let [form (read {:eof reader} reader)]
    (when-not (= form reader)
      (cons form (lazy-seq (form-seq reader))))))

(defmulti invoke-extension
          "An extension point for operations in config files."
          (fn [_ operation & _] operation))

(defn mapify
  "Turns a form read from a config file at `origin` into a map."
  [form origin]
  (cond
    (map? form) form
    (list? form) (let [[operation & args] form]
                   (invoke-extension origin operation args))
    :else (throw (ex-info (str "No support for mapifying form " (pr-str form))
                          {:reason :cant-mapify
                           :form   form
                           :origin origin}))))

(defn merge-config
  "Merges the configs read"
  [path config-forms]
  (reduce merge (map #(mapify % path) config-forms)))

(defn read-config
  "Returns a map read from `path`. Map will be generated by merging all
  forms found in file. Lists are interpreted as invocations of the
  `invoke-extension` multimethod, dispatched on the symbol in the
  first position.

  Example:

  (include \"something.edn\")
  ;; Comments are ignored
  {:foo :bar
   :bar 1234}

  Might yield:

  {:foo :bar
   :from-something 4321
   :bar 1234}"
  [path]
  (let [source (-> (io/reader path)
                   PushbackReader.)]
    (merge-config path (form-seq source))))

(defn path-relative-to
  "Given two paths `p1` and `p2`, returns a path that is the
  combination of them. If `p2` is absolute, `p2` is returned.
  Otherwise p1/p2 is returned."
  [^URL url other]
  (let [other-url (try (URL. other) (catch MalformedURLException e nil))
        resolved (if other-url
                   other-url
                   (URL. url (str "./" other)))]
    resolved))


;; E.g. (include "foo.edn") => {:some :map}
;; TODO: globbing support
(defmethod invoke-extension 'include
  [origin _ [path]]
  (read-config (path-relative-to origin path)))

(defmethod invoke-extension 'include-as
  [origin _ [key path]]
  {key (read-config (path-relative-to origin path))})

(defn config-name [env]
  (if env
    (str "config@" env ".edn")
    "config.edn"))

(defn init [& [env]]
  (read-config (io/resource (config-name env))))


(defmacro with-config
  [context-sym env & body]
  `(let [~context-sym (gorillalabs.config/init ~env)]
     ~@body))