(ns stash.api.stashapi
  (:gen-class)
  (:require [clojure.java.io :as io])                           ; File I/O support
  (:require [clj-http.client :as client])                       ; POST support
  (:require [cheshire.core :refer :all])                        ; JSON support
  (:import org.apache.commons.codec.binary.Hex                  ; bin2hex / hex2bin
           java.io.ByteArrayOutputStream                        ; Hashing Support
           (javax.crypto Mac)                                   ; HMAC calculations
           (javax.crypto KeyGenerator Cipher)                   ; Encrypt/Decrypt Routines
           (javax.crypto.spec SecretKeySpec IvParameterSpec)    ; Encrypt/Decrypt Support
           (java.security Key SecureRandom MessageDigest)       ; Encrypt/Decrypt & Hashing Support
  )
)
(defmacro apiversion [] (str "1.0"))  ; The API implementation version

(defn geturl
  "Gets the URL for the request"
  [urlIn endpoint]
  (str endpoint "/" urlIn)
)

(defn gettimestamp
  "Gets the current epoch timestamp"
  []
  (quot (System/currentTimeMillis) 1000)
)

; See https://gist.github.com/jhickner/2382543
(defn hmac
  "Calculate HMAC signature for given data."
  [^String key ^String data]
  (let [algo "HmacSHA256"
        signing-key (SecretKeySpec. (.getBytes key) algo)
        mac (doto (Mac/getInstance algo) (.init signing-key))]
    (format "%x" (BigInteger. 1 (.doFinal mac (.getBytes data))))))

(defn sha1 [io-factory]
  (let [bytes'
        (with-open [xin (io/input-stream io-factory)
                    xout (ByteArrayOutputStream.)]
          (io/copy xin xout)
          (.toByteArray xout))
        algorithm (MessageDigest/getInstance "SHA1")
        raw (.digest algorithm bytes')]
    (format "%032x" (BigInteger. 1 raw))))

(defn getsignature
  "Calculates the signature of the request and returns the value"
  [params options]
  ; url, api_version, api_id, api_timestamp, api_signature (blank) + all parameters (http_build_query or json_encoded string)
  (def sig (hmac (:pw options) (generate-string params)))
  (if (:verbose options)
  (do
    (println "Fields to sign: " params)
    (println "String to sign: " (generate-string params))
    (println "Signature: " sig)
  ))
  sig
)

(defn convertstringtoidentarray
  "Creates a map of the input string formatted like 'abc=1,def=\"string\",gh=My Test String,folderNames=My Home|vTest', and returns {:abc 1 :def \"string\" :gh \"My Test String\" :folderNames [\"My Home\" \"vTest\"]}"
  [options]
  (if (empty? options)
    #{}     ; If no options given, return empty map
    (do
      ; Process all the options
      (def tMap (apply merge (map #(hash-map (keyword (first %1)) (second %1)) (mapv #(clojure.string/split % #"=") (clojure.string/split options #",")))))
      ; Additional process for folderNames and destFolderNames to parse directory names by | and turn into vector
      (def tMap1
        (if (contains? tMap :folderNames)
          (update tMap :folderNames #(clojure.string/split % #"\|"))
          tMap
        )
      )
      (if (contains? tMap1 :destFolderNames)
        (update tMap1 :destFolderNames #(clojure.string/split % #"\|"))
        tMap1
      )
    )
  )
)

(defn sendrequest
  "Sends a request to the API endpoint"
  [^String url params options]
  (let [  requrl (geturl url (:endpoint options))
          fields (conj (hash-map :url requrl :api_id (:id options) :api_pw (:pw options) :api_timestamp (gettimestamp) :api_version (apiversion) :verbosity false) params)
          fieldswithsig (conj fields {:api_signature (getsignature fields options)})]
    (if (:verbose options)
      (do (println "Url: " requrl)
        (println "Fields: " fieldswithsig)
      )
    )
    ; See https://github.com/dakrone/clj-http#post
    (client/post requrl {:form-params fieldswithsig :content-type :json})
  )
)

(defn senddownloadrequest
  "Sends a request to download a file from the API endpoitn"
  [^String url params options]
  (println "Url: " url)
  (println "Params: " params)
  (println "Options: " options)
  (let [  requrl (geturl url (:endpoint options))
          fields (conj (hash-map :url requrl :api_id (:id options) :api_pw (:pw options) :api_timestamp (gettimestamp) :api_version (apiversion) :verbosity false) params)
          fieldswithsig (conj fields {:api_signature (getsignature fields options)})]
    (if (:verbose options)
      (do (println "Url: " requrl)
        (println "Fields: " fieldswithsig)
      )
    )
    ; See https://github.com/dakrone/clj-http#post
    (io/copy
      (:body (client/post requrl {:form-params fieldswithsig :content-type :json :as :stream}))
      (java.io.File. (:filename options)))
  )
)

(defn sendfilerequest
  "Uploads a file to the API endpoint"
[^String url params options]
  (println "Url: " url)
  (println "Params: " params)
  (println "Options: " options)
  (let [  requrl (geturl url (:endpoint options))
          fields (conj (hash-map :url requrl :api_id (:id options) :api_pw (:pw options) :api_timestamp (gettimestamp) :api_version (apiversion) :verbosity false) params)
          fieldswithsig (conj fields {:api_signature (getsignature fields options)})]
    (if (:verbose options)
      (do (println "Url: " requrl) (println "Fields: " fieldswithsig)))

    ; See https://github.com/dakrone/clj-http#post
    (client/post requrl {:multipart [{:name "params" :content (generate-string fieldswithsig)}
                                     {:name "file" :content (clojure.java.io/file (:filename options))}
    ]})
  )
)

(defn check-apiid-apipw
  "Checks for valid api_id and api_pw parameters have been set"
  [options]
  (cond
    (empty? (:id options)) (throw (IllegalArgumentException. "API_ID Must Be Specified"))
    (empty? (:pw options)) (throw (IllegalArgumentException. "API_PW Must Be Specified"))
    ; API ID can be an email address or a 32 character string
    (and (not= 32 (count (:id options))) (not (re-matches #".+\@.+\..+" (:id options)))) (throw (IllegalArgumentException. "API_ID Must Be 32 Characters or Email Address"))
    (not= 32 (count (:pw options))) (throw (IllegalArgumentException. "API_PW Must Be 32 Characters"))
  :else
    true
  )
)

(defn check-filekey-username
  "Checks for valid filekey and username parameters have been set"
  ([options checkfilekey checkusername]
  (cond
    (and checkfilekey (empty? (:fileKey options))) (throw (IllegalArgumentException. "fileKey Must Be Specified"))
    (and checkusername (empty? (:accountUsername options))) (throw (IllegalArgumentException. "accountUsername Must Be Specified"))
    (and checkfilekey (> 0 (count (:fileKey options)))) (throw (IllegalArgumentException. "fileKey Must Be Specified"))
    (and checkusername (> 0 (count (:accountUsername options)))) (throw (IllegalArgumentException. "accountUsername Must Be Specified"))
  :else
    true
  ))
  ([options]
  (check-filekey-username options true true))
)

(defn check-filename
  "Checks the filename is specified, and optionally, exists for get/put requests"
  [options mustExist]
  (cond
    (and (not mustExist) (contains? options :filename) (not(empty? (:filename options)))) true;
    (and mustExist (contains? options :filename) (not(empty? (:filename options))) (.exists (io/file (:filename options)))) true;
    :else
    (throw (IllegalArgumentException. "Filename Must Be Specified and/or File Must Exist"))
  )
)

(defn validateSourceParams
  "Checks for valid source parameters like fileId, fileName, folderId, and folderNames"
  [options folderOnly allowZeroIds]
  (if folderOnly
    ; Folder parameters only
    (cond
      (and folderOnly allowZeroIds (contains? options :folderId) (>= (Integer. (:folderId options)) -1)) true
      (and folderOnly (not allowZeroIds) (contains? options :folderId) (> (Integer. (:folderId options)) 0)) true
      (and folderOnly (contains? options :folderNames) (vector? (:folderNames options)) (> (count (:folderNames options)) 0)) true
      :else
      (throw (IllegalArgumentException. "Source Paramters Invalid - folderId or folderNames MUST be specified"))
    )
  ; Folder and File
    (cond
      (and allowZeroIds (contains? options :fileId) (>= (Integer. (:fileId options)) 0)) true
      (and (not allowZeroIds) (contains? options :fileId) (> (Integer. (:fileId options)) 0)) true
      (and (contains? options :fileName)
           (not (empty? (:fileName options)))
           (contains? options :folderId)
           ;(integer? (:folderId options))
           ;(> (:folderId options) 0)) true
           (> (Integer. (:folderId options)) 0)) true
      (and (contains? options :fileName)
           (not (empty? (:fileName options)))
           (contains? options :folderNames)
           (vector? (:folderNames options))
           (> (count (:folderNames options)) 0)) true
      :else
      (throw (IllegalArgumentException. "Source Parameters Invalid - fileId or fileName plus either folderId or folderNames MUST be specified"))
    )
  )
)

(defn validateDestParams
  "Checks for valid source parameters like destFolderId, destFolderNames, and destFileName"
  [options folderOnly nameOnly]
  (if folderOnly
    ; Folder parameters only
    (cond
      nameOnly (throw (IllegalArgumentException. "folderOnly and nameOnly cannot both be T"))
      (and (contains? options :destFolderId) (> (Integer. (:destFolderId options)) 0)) true
      (and (contains? options :destFolderNames) (vector? (:destFolderNames options)) (> (count (:destFolderNames options)) 0)) true
      :else
      (throw (IllegalArgumentException. "Destination Parameters Invalid - destFolderId or destFolderNames MUST be specified"))
    )
  ; NameOnly
    (cond
      (and nameOnly (contains? options :destFileName) (not (empty? (:destFileName options)))) true
      (and (contains? options :destFileName)
           (not (empty? (:destFileName options)))
           (contains? options :destFolderId)
           (> (Integer. (:destFolderId options)) 0)) true
      (and (contains? options :destFileName)
           (not (empty? (:destFileName options)))
           (contains? options :destFolderNames)
           (vector? (:destFolderNames options))
           (> (count (:destFolderNames options)) 0)) true
      :else
      (throw (IllegalArgumentException. "Destination Parameters Invalid - destFileName plus either destFolderId or destFolderNames MUST be specified"))
    )
  )
)

(defn validateOverwriteParams
  "Checks for valid overwrite parameters like overwriteFile and overwriteFileId"
  [params]
  (cond
    (not (contains? params :overwriteFile)) true    ; No overwriteFile = no need to check further
    (and (contains? params :overwriteFile) (or (> (Integer. (:overwriteFile params)) 1) (< (Integer. (:overwriteFile params)) 0))) (throw (IllegalArgumentException. "Invalid overwriteFile value"))
    (and (contains? params :overwriteFile) (= (Integer. (:overwriteFile params)) 0)) true
    (and (contains? params :overwriteFile)
         (= (Integer. (:overwriteFile params)) 1)
         (not (contains? params :overwriteFileId))) (throw (IllegalArgumentException. "overwriteFileId parameter must be specified with overwriteFile"))
    (and (contains? params :overwriteFile)
         (= (Integer. (:overwriteFile params)) 1)
         (contains? params :overwriteFileId)
         (< (Integer. (:overwriteFileId params)) 1)) (throw (IllegalArgumentException. "Invalid value for overwriteFileId"))
    (and (contains? params :overwriteFile)
         (= (Integer. (:overwriteFile params)) 1)
         (contains? params :overwriteFileId)
         (>= (Integer. (:overwriteFileId params)) 1)) true
    :else
    (throw (IllegalArgumentException. "Invalid overwriteFile and/or overwriteFileId parameters"))
  )
)

(defn random-bytes
  "Returns a random byte array of the specified size."
  ^bytes
  [size]
  (let [buffer (byte-array size)]
    (.nextBytes (SecureRandom.) buffer)
    buffer))

; Encrypts a byte-array 'ba' with key 'key' and initialization vector 'ivIn'; for aes-cbc, the IV must be 16 bytes
(defn encrypt
  [ba key ivIn]
  (let [iv  (IvParameterSpec. (byte-array (mapv #(if (> % 127) (- % 256) %) ivIn)))
        spec   (SecretKeySpec. (byte-array 32 key) "AES")
        cipher (Cipher/getInstance "AES/CBC/PKCS5Padding")
       ]
       (.init cipher Cipher/ENCRYPT_MODE spec iv)
       (doall (.doFinal cipher ba))))

; Wrapper function to encrypt a string 's' with key 'key'; generates a random IV of 16 bytes
(defn encryptstring
  [^String s ^String key]
  (let [key (.getBytes key)
        ivIn (random-bytes 16)
        ct (encrypt (.getBytes s) key ivIn)
        ctHex (Hex/encodeHexString ct)
        ivHex (Hex/encodeHexString ivIn)
        ]
    (apply str (concat ivHex ctHex))
  )
)

; Decrypts a string 's', given a key 'key' and initialization vector 'ivIn'
(defn decrypt
  [s key ivIn]
  (let [iv  (IvParameterSpec. (byte-array (mapv #(if (> % 127) (- % 256) %) ivIn)))
        spec   (SecretKeySpec. (byte-array 32 key) "AES")
        cipher (Cipher/getInstance "AES/CBC/PKCS5Padding")
       ]
       (.init cipher Cipher/DECRYPT_MODE spec iv)
       (.doFinal cipher s)))

; Wrapper function to decrypt a string 's' with key 'key'
(defn decryptstring
  [^String s ^String key]
  (let [key (.getBytes key)
        ivHex (subs s 0 32)
        ctHex (subs s 32)
        ivIn (Hex/decodeHex ivHex)
        ctIn (Hex/decodeHex ctHex)
        pt (decrypt ctIn key ivIn)
      ]
      (String. pt "UTF-8")
  )
)

; ---------------------------------------------------------------------------------------------------------------------
; API Request Functions
; These functions build and send the corresponding API request
; ---------------------------------------------------------------------------------------------------------------------
(defn loopback-reqs
  "Checks the requirements for calling the loopback request"
  [options]
  (check-apiid-apipw options)
  true
)

(defn loopback
  "Sends a loopback request to the API endpoint"
  [options]
  (loopback-reqs options)
  (if (:verbose options)    ; If verbosity set, return entire response; otherwise just the response body
    (sendrequest "api2/file/testloopback" (convertstringtoidentarray (:srcId options)) options)
    (:body (sendrequest "api2/file/testloopback" (convertstringtoidentarray (:srcId options)) options))
  )
)

(defn putfile-reqs
  "Checks the requirements for calling the putfile request"
  [params]
  (validateDestParams params true false)
  (validateOverwriteParams params)
  (check-filekey-username params)
  (check-filename params false)
)

(defn putfile
  "Sends a putfile request to the API endpoint"
  [options]
  ( let [params (conj {:fileKey (encryptstring (:filekey options) (:pw options)) :accountUsername (:username options) :filename (:filename options)} (convertstringtoidentarray (:dstId options)))]
    (putfile-reqs params)

    ; Check for file exists, and error if not overwriting
    ; *** This step may not be necessary if the server checks for overwrite on upload anyway ***

    (if (:verbose options)    ; If verbosity set, return entire response; otherwise just the response body
      (sendfilerequest "api2/file/write" params options)
      (:body (sendfilerequest "api2/file/write" params options))
    )
    ; ToDo
    ; Check response and return 200, OK, filename if success, otherwise, delete local file and return error message, code or -1 or 0
  )
)

(defn getfile-reqs
  "Checks the requirements for calling the getfile request"
  [params]
  (println "params: " params)
  (validateSourceParams params false false)
  (check-filekey-username params)
)

(defn getfile
  "Sends a getfile request to the API endpoint"
  [options]
  ( let [params (conj {:fileKey (encryptstring (:filekey options) (:pw options)) :accountUsername (:username options)} (convertstringtoidentarray (:srcId options)))]
    (getfile-reqs params)
    (if (:verbose options)    ; If verbosity set, return entire response; otherwise just the response body
      (senddownloadrequest "api2/file/read" params options)
      (:body (senddownloadrequest "api2/file/read" params options))
    )
    ; ToDo
    ; Check response and return 200, OK, filename if success, otherwise, delete local file and return error message, code or -1 or 0
  )
)

(defn copyfile-reqs
  "Checks the requirements for calling the copyfile request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn copyfile
  "Sends a copyfile request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn renamefile-reqs
  "Checks the requirements for calling the renamefile request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn renamefile
  "Sends a renamefile request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn movefile-reqs
  "Checks the requirements for calling the movefile request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn movefile
  "Sends a movefile request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn deletefile-reqs
  "Checks the requirements for calling the deletefile request"
  [params]
  (validateSourceParams params false false)
)

(defn deletefile
  "Sends a deletefile request to the API endpoint"
  [options]
  ( let [params (convertstringtoidentarray (:srcId options))]
    (deletefile-reqs params)
    (if (:verbose options)    ; If verbosity set, return entire response; otherwise just the response body
      (sendrequest "api2/file/delete" params options)
      (:body (sendrequest "api2/file/delete" params options))
    )
  )
)

(defn listall-reqs
  "Checks the requirements for calling the listall request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn listall
  "Sends a listall request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn listfiles-reqs
  "Checks the requirements for calling the listfiles request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn listfiles
  "Sends a listfiles request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn listsffiles-reqs
  "Checks the requirements for calling the listsffiles request"
  [options]
  (throw (Exception. "Not Implemented"))
)

; List Special Folder files
(defn listsffiles
  "Sends a listsffiles request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn listfolders-reqs
  "Checks the requirements for calling the listfolders request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn listfolders
  "Sends a listfolders request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn getfolderid-reqs
  "Checks the requirements for calling the getfolderid request"
  [options]
  (validateSourceParams options true false)
)

(defn getfolderid
  "Sends a getfolderid request to the API endpoint"
  [options]
  ( let [params (convertstringtoidentarray (:srcId options))]
    (getfolderid-reqs params)
    (if (:verbose options)    ; If verbosity set, return entire response; otherwise just the response body
      (sendrequest "api2/file/getfolderid" params options)
      (:body (sendrequest "api2/file/getfolderid" params options))
    )
  )
)

(defn setfilelock-reqs
  "Checks the requirements for calling the setfilelock request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn setfilelock
  "Sends a setfilelock request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn getfilelock-reqs
  "Checks the requirements for calling the getfilelock request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn getfilelock
  "Sends a getfilelock request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn clearfilelock-reqs
  "Checks the requirements for calling the clearfilelock request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn clearfilelock
  "Sends a clearfilelock request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn gettags-reqs
  "Checks the requirements for calling the gettags request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn gettags
  "Sends a gettags request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn settags-reqs
  "Checks the requirements for calling the settags request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn settags
  "Sends a settags request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn addtag-reqs
  "Checks the requirements for calling the addtag request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn addtag
  "Sends a addtag request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn deletetag-reqs
  "Checks the requirements for calling the deletetag request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn deletetag
  "Sends a deletetag request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn createdirectory-reqs
  "Checks the requirements for calling the createdirectory request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn createdirectory
  "Sends a createdirectory request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn renamedirectory-reqs
  "Checks the requirements for calling the renamedirectory request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn renamedirectory
  "Sends a renamedirectory request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn movedirectory-reqs
  "Checks the requirements for calling the movedirectory request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn movedirectory
  "Sends a movedirectory request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn copydirectory-reqs
  "Checks the requirements for calling the copydirectory request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn copydirectory
  "Sends a copydirectory request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn deletedirectory-reqs
  "Checks the requirements for calling the deletedirectory request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn deletedirectory
  "Sends a deletedirectory request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn getfileinfo-reqs
  "Checks the requirements for calling the getfileinfo request"
  [params]
  (validateSourceParams params false false)
)

(defn getfileinfo
  "Sends a getfileinfo request to the API endpoint"
  [options]
  ( let [params (convertstringtoidentarray (:srcId options))]
    (getfileinfo-reqs params)
    (if (:verbose options)    ; If verbosity set, return entire response; otherwise just the response body
      (sendrequest "api2/file/getfileinfo" params options)
      (:body (sendrequest "api2/file/getfileinfo" params options))
    )
  )
)

(defn getfolderinfo-reqs
  "Checks the requirements for calling the getfolderinfo request"
  [params]
  (validateSourceParams params true false)
)

(defn getfolderinfo
  "Sends a getfolderinfo request to the API endpoint"
  [options]
  ( let [params (convertstringtoidentarray (:srcId options))]
    (getfolderinfo-reqs params)
    (if (:verbose options)    ; If verbosity set, return entire response; otherwise just the response body
      (sendrequest "api2/file/getfolderinfo" params options)
      (:body (sendrequest "api2/file/getfolderinfo" params options))
    )
  )
)

(defn getsyncinfo-reqs
  "Checks the requirements for calling the getsyncinfo request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn getsyncinfo
  "Sends a getsyncinfo request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn getvaultinfo-reqs
  "Checks the requirements for calling the getvaultinfo request"
  [options]
  ; Only the API ID and API PW are required
  (check-apiid-apipw options)
)

(defn getvaultinfo
  "Sends a getvaultinfo request to the API endpoint; returns JSON"
  [options]
  (getvaultinfo-reqs options)
  (if (:verbose options)    ; If verbosity set, return entire response; otherwise just the response body
    (sendrequest "api2/file/getvaultinfo" {} options)
    (:body (sendrequest "api2/file/getvaultinfo" {} options))
  )
)

(defn checkcreds-reqs
  "Checks the requirements for calling the checkcreds request"
  [options]
  ; Requires the api_id, api_pw, fileKey and username parameters
  (check-apiid-apipw options)
  (check-filekey-username options)
  true
)

(defn checkcreds
  "Sends a checkcreds request to the API endpoint"
  [options]
  (checkcreds-reqs options)
  (if (:verbose options)    ; If verbosity set, return entire response; otherwise just the response body
    (sendrequest "api2/auth/checkcreds" {:fileKey (encryptstring (:filekey options) (:pw options)) :accountUsername (:username options)} options)
    (:body (sendrequest "api2/auth/checkcreds" {:fileKey (encryptstring (:filekey options) (:pw options)) :accountUsername (:username options)} options))
  )
)

(defn checkcredsad-reqs
  "Checks the requirements for calling the checkcredsad request"
  [options]
  ; Requires the api_id, api_pw, fileKey and username parameters
  (check-apiid-apipw options)
  (check-filekey-username options true false)
)

(defn checkcredsad
  "Sends a checkcredsad request to the API endpoint"
  [options]
  (checkcredsad-reqs options)
  (if (:verbose options)    ; If verbosity set, return entire response; otherwise just the response body
    (sendrequest "api2/auth/adauth" {:fileKey (encryptstring (:filekey options) (:pw options)) :accountUsername (:username options)} options)
    (:body (sendrequest "api2/auth/adauth" {:fileKey (encryptstring (:filekey options) (:pw options)) :accountUsername (:username options)} options))
  )
)

(defn checkvaultconnection-reqs
  "Checks the requirements for calling the checkvaultconnection request"
  [options]
  (loopback-reqs options)
  true
)

(defn checkvaultconnection
  "Sends a checkvaultconnection request to the API endpoint"
  [options]
  (checkvaultconnection-reqs options)
  (let [response (parse-string (loopback options) true)]
    (if (= 200 (:code response))
      true
      false
    )
  )
)

(defn isvaliduser-reqs
  "Checks the requirements for calling the isvaliduser request"
  [options]
  (check-filekey-username options false true)
)

(defn isvaliduser
  "Sends a isvaliduser request to the API endpoint"
  [options]
  (isvaliduser-reqs options)
  (if (:verbose options)    ; If verbosity set, return entire response; otherwise just the response body
    (sendrequest "api2/auth/isvaliduser" {:accountUsername (:username options)} options)
    (:body (sendrequest "api2/auth/isvaliduser" {:accountUsername (:username options)} options))
  )
)

(defn setpermissions-reqs
  "Checks the requirements for calling the setpermissions request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn setpermissions
  "Sends a setpermissions request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn checkpermissions-reqs
  "Checks the requirements for calling the checkpermissions request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn checkpermissions
  "Sends a checkpermissions request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn listversions-reqs
  "Checks the requirements for calling the listversions request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn listversions
  "Sends a listversions request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn readversion-reqs
  "Checks the requirements for calling the readversion request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn readversion
  "Sends a readversion request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn restoreversion-reqs
  "Checks the requirements for calling the restoreversion request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn restoreversion
  "Sends a restoreversion request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn deleteversion-reqs
  "Checks the requirements for calling the deleteversion request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn deleteversion
  "Sends a deleteversion request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberasestore-reqs
  "Checks the requirements for calling the weberasestore request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberasestore
  "Sends a weberasestore request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberasedownload-reqs
  "Checks the requirements for calling the weberasedownload request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberasedownload
  "Sends a weberasedownload request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberaseretrieve-reqs
  "Checks the requirements for calling the weberaseretrieve request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberaseretrieve
  "Sends a weberaseretrieve request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberasepolling-reqs
  "Checks the requirements for calling the weberasepolling request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberasepolling
  "Sends a weberasepolling request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberaseupdate-reqs
  "Checks the requirements for calling the weberaseupdate request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberaseupdate
  "Sends a weberaseupdate request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberasetoken-reqs
  "Checks the requirements for calling the weberasetoken request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberasetoken
  "Sends a weberasetoken request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberaseonetimecode-reqs
  "Checks the requirements for calling the weberaseonetimecode request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberaseonetimecode
  "Sends a weberaseonetimecode request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberasedelete-reqs
  "Checks the requirements for calling the weberasedelete request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberasedelete
  "Sends a weberasedelete request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberaseprojectlist-reqs
  "Checks the requirements for calling the weberaseprojectlist request"
  [options]
  (throw (Exception. "Not Implemented"))
)

(defn weberaseprojectlist
  "Sends a weberaseprojectlist request to the API endpoint"
  [options]
  (throw (Exception. "Not Implemented"))
)
