(ns api.remote
  (:require
   [api.util :as util]
   [api.http :as http]
   [reagent.core :as reagent]
   [api.crypto :as crypto]
   [cljs.reader :refer [read-string]]
   [cognitect.transit :as t]
   [re-frame.core :refer [reg-event-db path reg-sub dispatch dispatch-sync subscribe]]
   [cljs.core.async :as async :refer [chan close! put! <! timeout]]
   [cljs.spec.gen.alpha])
  (:require-macros [cljs.core.async.macros :refer [go alt!]]
                   [cljs.spec.gen.alpha :refer [dynaload]]))

(def endpoint (atom nil))

(defn support-transit? []
  (or (not (exists? js/window))
      (not (aget js/window "no_transit_support"))))

(def api-caller (dynaload 'api.server/api-caller))

(defn GET [url]
  (go (let [{:keys [status data error]} (<! (http/GET-PLAIN url))]
        (if (= status :ok)
          data
          (js/Error. error)))))

(defn api-background
  ([path] (api-background path nil))
  ([path & params]
   (dispatch-sync [:loading])
   (go
     (let [r (@api-caller path (if (support-transit?)
                                 (t/write (t/writer :json) (vec params))
                                 (pr-str (vec params))))
           {:keys [status body] :as all} (if (util/chan? r) (<! r) r)
           r (if (= status 200)
               (if (support-transit?)
                 (t/read (t/reader :json) (.toString (crypto/unzip body) "utf-8"))
                 (read-string body))
               (let [body (if (support-transit?)
                            (t/read (t/reader :json) body)
                            (read-string body))]
                 (when (= "Invalid token" body)
                   (dispatch-sync [:logout]))
                 nil))]
       (dispatch [:unloading])
       r))))

(defn api
  ([path] (api path nil))
  ([path & params]
   (go (<! (apply api-background path params)))))
