(ns api.macros
  (:require [cljs.analyzer :as ana]
            [cljs.analyzer.api :as ana-api]
            [cljs.spec.alpha :as s]
            [clojure.string :as string]
            [cemerick.url :refer [url url-encode url-decode]]
            #?(:cljs [cljs.core.async :as async])
            #?(:cljs [cljs.spec.gen.alpha :as gen]))
  #?(:cljs
     (:require-macros [cljs.core.async.macros :refer [go]])))

#?(:clj
   (defmacro defapi [n]
     (let [nn (str "/" (name n))]
       `(defmethod api.server/dispatch ~nn [request#]
          (let [request# (cljs.core/js->clj request# :keywordize-keys true)
                body# (:body request#)
                body# (if (.startsWith body# "body=")
                        (string/replace (subs body# 5) "+" "%20")
                        body#)
                body# (api.server/read-transit (url-decode body#))
                token# (:value ((:cookies request#) "token"))]
            (cljs.core.async.macros/go
              (try
                (let [user# (jwt/verify token#)]
                  (api.aws/capture
                   "user"
                   {:merchant (:merchant user#)
                    :user (or (:terminal user#) (:email user#))})
                  (if (or (:terminal user#) (:email user#))
                    (try
                      (let [r (binding [api.core/*request* request#]
                                (apply ~n user# body#))]
                        (api.server/transit-response
                         :ok (if (api.util/chan? r) (glossop.core/<? r) r)))
                      (catch js/Error e#
                        (glossop.core/<? (api.server/log-error user# e#))
                        (api.server/transit-response
                         :er
                         {:status :er
                          :details (or (:error (ex-data e#)) :internal-error)
                          :msg (.-message e#)})))
                    (do
                      (glossop.core/<? (api.server/log-error
                                        nil (js/Error. (str "Invalid token " (pr-str user#)))))
                      (api.server/transit-response :er "Invalid token"))))
                (catch js/Error e#
                  (glossop.core/<? (api.server/log-error nil e#))
                  (api.server/transit-response :er "Invalid token")))))))))

#?(:clj
   (defmacro defapiopen [n]
     (let [nn (str "/" (name n))]
       `(defmethod api.server/dispatch ~nn [request#]
          (let [request# (cljs.core/js->clj request# :keywordize-keys true)
                body# (:body request#)
                body# (if (.startsWith body# "body=")
                        (string/replace (subs body# 5) "+" "%20")
                        body#)
                body# (api.server/read-transit (url-decode body#))]
            (cljs.core.async.macros/go
              (try
                (try
                  (let [r (binding [api.core/*request* request#]
                            (apply ~n body#))]
                    (api.server/transit-response
                     :ok (if (api.util/chan? r) (glossop.core/<? r) r)))
                  (catch js/Error e#
                    (glossop.core/<? (api.server/log-error nil e#))
                    (api.server/transit-response
                     :er
                     {:status :er
                      :details (or (:error (ex-data e#)) :internal-error)
                      :msg (.-message e#)})))
                (catch js/Error e#
                  (glossop.core/<? (api.server/log-error nil e#))
                  (api.server/transit-response :er "Invalid token")))))))))

#?(:cljs
   (defn- make-check-result
     "Builds spec result map."
     [check-sym spec test-check-ret]
     (merge {:spec spec
             :clojure.test.check/ret test-check-ret}
            (when check-sym
              {:sym check-sym})
            (when-let [result (-> test-check-ret :result)]
              (when-not (true? result) {:failure result}))
            (when-let [shrunk (-> test-check-ret :shrunk)]
              {:failure (:result shrunk)}))))

#?(:cljs
   (defn- explain-check
     [args spec v role]
     (ex-info
      "Specification-based check failed"
      (when-not (s/valid? spec v nil)
        (assoc (s/explain-data* spec [role] [] [] v)
               ::args args
               ::val v
               ::s/failure :check-failed)))))

(defn- fn-spec-name?
  [s]
  (symbol? s))

(defn- collectionize
  [x]
  (if (symbol? x)
    (list x)
    x))

(defn checkable-syms*
  ([]
   (checkable-syms* nil))
  ([opts]
   (reduce into #{}
     [(filter fn-spec-name? (keys @s/registry-ref))
      (keys (:spec opts))])))

#?(:cljs
   (defn- check-call [events f specs args]
     (let [cargs (when (:args specs) (s/conform (:args specs) args))]
       (if (= cargs ::s/invalid)
         (explain-check args (:args specs) args :args)
         (let [init-ret (when-let [i (:initialize events)] (i f args))
               ret (apply f args)
               ret (if (api.util/chan? ret)
                     (let [ret-atom (atom nil)
                           ret-done (atom false)]
                       (go
                         (reset! ret-atom (async/<! ret))
                         (reset! ret-done true))
                       (.loopWhile (cljs.nodejs/require "deasync")
                                   (fn [] (not @ret-done)))
                       @ret-atom)
                     ret)
               _ (when-let [c (:cleanup events)] (c init-ret f args ret))
               cret (when (:ret specs) (s/conform (:ret specs) ret))]
           (if (= cret ::s/invalid)
             (explain-check args (:ret specs) ret :ret
              )
             (if (and (:args specs) (:ret specs) (:fn specs))
               (if (s/valid? (:fn specs) {:args cargs :ret cret})
                 true
                 (explain-check args (:fn specs) {:args cargs :ret cret} :fn))
               true)))))))

#?(:cljs
   (defn- quick-check
     [f specs events {gen :gen opts :clojure.test.check/opts}]
     (let [{:keys [num-tests] :or {num-tests 1000}} opts
           g (try (s/gen (:args specs) gen) (catch js/Error t t))]
       (if (instance? js/Error g)
         {:result g}
         (let [init-ret (when-let [i (:initialize-once events)] (i))
               prop (cljs.spec.gen.alpha/for-all*
                     [g] #(check-call events f specs %))
               r (apply cljs.spec.gen.alpha/quick-check num-tests prop
                        (mapcat identity opts))]
           (when-let [i (:cleanup-once events)] (i init-ret))
           r)))))

#?(:cljs
   (defn my-check-fn [logger sym name v events opts instrument]
     (logger {:stage :before :sym sym})
     (let [ch (cljs.core.async/chan 1)]
       (cljs.core.async.macros/go
         (try
           (let [s        name
                 spec (s/get-spec v)
                 re-inst? (and v (instrument false) true)
                 f         @v
                 r (cond
                     (nil? f)
                     {:failure (ex-info "No fn to spec" {::s/failure :no-fn})
                      :sym s
                      :spec spec}

                     (:args spec)
                     (let [tcret (quick-check f spec events opts)]
                       (make-check-result s spec tcret))

                     :default
                     {:failure (ex-info "No :args spec" {::s/failure :no-args-spec})
                      :sym s :spec spec})]
             (logger {:stage :after :sym sym :data r})
             (try
               (cljs.core.async/put! ch r)
               (finally
                 (when re-inst? (instrument true)))))
           (catch js/Error e
             (cljs.core.async/put! ch e)
             (logger {:stage :error :sym sym :data (.-message e)}))))
       ch)))

#?(:clj
   (defmacro log-check
     ([opts logger events]
      `(log-check '~(checkable-syms*) ~opts ~logger ~events))
     ([sym-or-syms opts logger {:keys [skip only] :as events}]
      (let [skip (or skip #{})
            sym-or-syms (eval sym-or-syms)
            opts-sym    (gensym "opts")
            events-sym    (gensym "events")]
        `(cljs.core.async.macros/go
           (let [~opts-sym ~opts
                 ~events-sym ~events]
             [~@(->>
                 (collectionize sym-or-syms)
                 (filter (checkable-syms* opts))
                 (filter #(or (nil? only) (= % only)))
                 (remove skip)
                 (map
                  (fn [sym]
                    (let [{:keys [name] :as v} (ana-api/resolve &env sym)
                          v (when v `(var ~name))]
                      `(async/<!
                        (my-check-fn
                         ~logger '~sym
                         ~name ~v ~events-sym ~opts-sym
                         #(if %
                            (cljs.spec.test.alpha/instrument '~name)
                            (seq (cljs.spec.test.alpha/unstrument '~name)))))))))]))))))
#?(:clj
   (defmacro public-vars [ns]
     (let [ns-str (name ns)]
       `(set
         (for [sym# (quote ~(sort (keys (ana-api/ns-publics ns))))]
           (symbol ~ns-str (name sym#)))))))

#?(:clj
   (defmacro spec-syms [ns]
     (let [ns (name ns)]
       `(set (filter #(= '~ns (namespace %)) '~(checkable-syms*))))))

#?(:clj
   (defmacro my-instrument [opts]
     `(cljs.spec.test.alpha/instrument
       '[~@(#?(:clj  s/speced-vars
               :cljs cljs.spec.alpha$macros/speced-vars))] ~opts)))
