(ns api.lambda-logs
  (:require [cljs.nodejs :as nodejs]
            [cljs.reader :as reader]
            [api.util :as util]
            [cljs.core.async :as async]
            [api.aws :refer [AWS]]
            [cljs.pprint :as pp]
            [api.time :as time]
            [goog.i18n.DateTimeFormat :as dtf]
            [clojure.string :as string])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(def logs-class (aget AWS "CloudWatchLogs"))
(def logs (new logs-class #js {:region "us-east-1"}))
(def filter-log-events-call (aget logs "filterLogEvents"))
(def get-log-events-call (aget logs "getLogEvents"))
(def put-log-events-call (aget logs "putLogEvents"))
(def create-log-group-call (aget logs "createLogGroup"))
(def create-log-stream-call (aget logs "createLogStream"))

(defn create-log-group [name callback error]
  (let [params (clj->js
                {:logGroupName name})]
    (.call create-log-group-call
           logs
           params
           (fn [er data]
             (if er
               (error er)
               (callback data))))))

(defn create-log-stream [name stream callback error]
  (let [params (clj->js
                {:logGroupName name
                 :logStreamName stream})]
    (.call create-log-stream-call
           logs
           params
           (fn [er data]
             (if er
               (error er)
               (callback data))))))

(defn put-log-events
  ([group stream events callback error]
   (put-log-events group stream events nil callback error))
  ([group stream events sequence-token callback error]
   (let [params (clj->js
                 {:logGroupName group
                  :sequenceToken sequence-token
                  :logStreamName stream
                  :logEvents events})]
     (.call put-log-events-call logs params
      (fn [er data]
        (if er
          (let [code (aget er "code")
                message (aget er "message")]
            (if (= code "ResourceNotFoundException")
              (let [resource (get (string/split message " ") 3)
                    success (fn [r]
                              (put-log-events
                               group stream events sequence-token
                               callback error))]
                (println "Missing resource:" resource)
                (case resource
                  "group" (create-log-group group success error)
                  "stream" (create-log-stream group stream success error)
                  (error er)))
              (if (or (= "DataAlreadyAcceptedException" code)
                      (= "InvalidSequenceTokenException" code))
                (do
                  (println "RETRYING..." message)
                  (let [sequence (last (string/split message " "))]
                    (put-log-events group stream events sequence
                                    callback error)))
                (error er))))
          (callback data)))))))

(defn filter-log-events [group start end filter token callback error]
  (let [start (.getTime start)
        end (.getTime end)
        params (clj->js
                {:logGroupName group
                 :filterPattern filter
                 :nextToken token
                 :startTime start
                 :endTime end})]
    (.call filter-log-events-call
           logs
           params
           (fn [er data]
             (if er
               (error er)
               (callback data))))))

(defn <!filter-log-events
  ([group start end filter] (<!filter-log-events group start end filter nil))
  ([group start end filter token]
   (let [ch (async/chan 1)]
     (filter-log-events
      group start end filter token
      (fn [data]
        (let [t (aget data "nextToken")
              data (:events (js->clj data :keywordize-keys true))]
          (go (async/>!
               ch
               (if t
                 (into data (<! (<!filter-log-events group start end filter t)))
                 data)))))
      (fn [er] (async/>! ch er)))
     ch)))

(def ses-class (aget AWS "SES"))
(def ses (new ses-class #js {:region "us-east-1"}))
(def sendEmail (aget ses "sendEmail"))

(def red "\033[0;31m")
(def blue "\033[94m")
(def green "\033[1;31;42m")
(def yellow "\033[93m")
(def nc "\033[0m")
(def underline "\033[4")

(defn local? []
  (= "local" (aget (aget js/process "env") "ENVIRONMENT")))

(defn format-date [date-format date]
  (.format (goog.i18n.DateTimeFormat. date-format) (js/Date. date)))

(defn load-events [group-name since-minutes token callback error]
  (let [start (- (.getTime (js/Date.)) (* 1000 60 since-minutes))
        params (clj->js
                {:logGroupName group-name
                 :filterPattern "Exception"
                 :nextToken token
                 :startTime start
                 :endTime (+ start (* 1000 60 60 24))})]
    (.call filter-log-events-call
           logs
           params
           (fn [er data]
             (if er
               (error er)
               (callback data))))))

(defn parse [data]
  (vec
   (sort-by
    :date
    (for [e data]
      (try
        (let [msg (:message e)
              i (+ 10 (.indexOf msg "Exception:"))
              title (subs msg 0 i)
              msg (subs msg (inc i))
              data (reader/read-string (clojure.string/replace msg "\n" ""))
              title (subs title 0 (.indexOf title \tab))
              date (reader/read-string (str "#inst \"" title "\""))
              title (if (:line data)
                      (str (:message data) "@" (:line data) ":" (:column data))
                      (:message data))]
          (let [title-first (clojure.string/starts-with? (first (:stack data)) "Message")]
            (if (or title-first
                    (clojure.string/starts-with? (second (:stack data)) "Message"))
              {:date date
               :title (subs ((if title-first first second) (:stack data)) 9)
               :data (if (map? data)
                       (assoc data
                              :params (first (:stack data))
                              :stack (update-in (vec (butlast (next ((if title-first identity next) (:stack data)))))
                                                [0] #(subs % (count "StackTrace :"))))
                       {:raw data})}
              {:date date
               :title title
               :data (if (map? data) data {:raw data})})))
        (catch js/Error _
          (let [msg (:message e)
                msg (clojure.string/replace msg #"[+]" " ")
                i (+ 10 (.indexOf msg "Exception:"))
                title (subs msg 0 i)]
            {:date (js/Date.)
             :title (subs msg (inc i))
             :data {}})))))))

(defn resolve-fun [s l]
  (when l
    (if (not (= -1 (.indexOf l "api/")))
      (-> (last (clojure.string/split s #"api/" 2))
          (clojure.string/replace #"[/]" "."))
      (if (not (= -1 (.indexOf l "target/")))
        (-> (last (clojure.string/split s #"target/api/" 2))
            (clojure.string/replace #"[/]" "."))
        l))))

(defn console-output [data]
  (doseq [{:keys [title data]} data]
    (println (str underline green title nc))
    (doseq [[k v] (dissoc data :message :line :column)]
      (println (str blue k nc))
      (cond
        (vector? v)
        (do
          (doseq [s v]
            (try
              (let [[e l] (clojure.string/split s #"\@" 2)
                    l (resolve-fun s l)]
                (println "  " red e (when l (str yellow l)) nc))
              (catch js/Error e
                (println "  " red s nc))))
          (println ""))
        :else
        (pp/pprint v)))))

(defn html-output [data]
  (with-out-str
    (do
      (println "<div style='background-color:rgba(0,0,0,0.9);text-align:left;font-size:15px;padding:5px'>")
      (doseq [{:keys [title data]} data]
        (println (str "<div style='background-color:#0D0;color:#D00;padding:5px;font-weight:bold;padding:5px'>" title "</div>"))
        (doseq [[k v] (dissoc data :message :line :column)]
          (println (str "<div style='color:blue'>"  k "</div>"))
          (cond
            (vector? v)
            (do
              (doseq [s v]
                (let [[e l] (clojure.string/split s #"\@" 2)
                      l (resolve-fun s l)]
                  (println "<div style='margin-left:30px;color:red'>" e "<span style='color:yellow'>" l "</span></div>"))))
            :else
            (do
              (println "<div style='color:white;margin-left:30px'>")
              (let [all (with-out-str (pp/pprint v))]
                (println (clojure.string/replace all #"\n" "<br>")))
              (println "</div>")))))
      (println "</div>"))))

#_(defn process [data done]
  (let [data (parse (:events (js->clj data :keywordize-keys true)))]
    (console-output data)
    (when-not (local?)
      (.call sendEmail ses
             (clj->js
              {:Destination {:ToAddresses ["gal@zuldi.com"]}
               :Source "info@zuldi.com"
               :Message {:Subject
                         {:Data
                          (str "Zuldi Exception for "
                               (:merchant/name (:merchant (:data (first data)))))
                          :Charset "utf-8"}
                         :Body {:Html {:Data (html-output data)
                                       :Charset "utf-8"}}}})
             (fn [er data]
               (if er
                 (.log js/console er)
                 (.log js/console data))
               (done))))))

(defn load-since [token group-name since page]
  (let [ch (async/chan 1)]
    (load-events
     group-name
     since
     token
     (fn [data]
       (let [t (aget data "nextToken")
             data (parse (:events (js->clj data :keywordize-keys true)))]
         (go (async/>!
              ch
              (if t
                (into data (<! (load-since t group-name since (inc page))))
                data)))))
     (fn [er]
       (.error js/console er)))
    ch))
