(ns api.dynamo
  (:require [hildebrand.core :as db]
            [api.util :refer [local?]]
            [eulalie.util :refer [env!]]
            [clojure.set :as cset]
            [eulalie.creds :as creds]
            [api.ses :as ses]
            [api.aws :as aws]
            [cljs.core.async :as async :refer [<!]]
            [glossop.core :as g :refer-macros [go-catching <?]
             :refer [error?]])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defonce port (atom 8001))

(defn creds []
  (if (local?)
    {:endpoint (str "http://localhost:" @port)}
    (creds/env)))

(defn make-key []
  (str (.getTime (js/Date.)) "-" (random-uuid)))

;; TODO EMAIL
(defn wrap-ex [ch params]
  (let [table (second params)
        table? (keyword? table)]
    (aws/capture-node
     (str "dynamo" (if table? (str ":" (name table))))
     (fn [subsegment]
       (go-catching
        (let [r (<! ch)]
          (aws/add-operation subsegment (first params))
          (aws/add-metadata subsegment "query" (pr-str params))
          (when (error? r)
            (aws/add-error subsegment (js/Error. (pr-str r)) true)
            (println "DYNAMO ERROR" (pr-str r) (pr-str params))
            #_(when-not (local?)
              (<! (ses/send-email!
                   ["dynamo@zuldi.com"]
                   (pr-str [r params])
                   {:encoding "UTF-8" :data "DynamoDB Error"}
                   "no-reply@zuldi.com"))))
          (aws/close subsegment)
          r))))))

(defn query [table where dyn & [page]]
  (wrap-ex
   (db/query! (creds) table where dyn (merge {:consistent true} page))
   [:query table where dyn (merge {:consistent true} page)]))

(defn query-with-creds [creds table where dyn & [page]]
  (wrap-ex
   (db/query! creds table where dyn (merge {:consistent true} page))
   [:query-with-creds table where dyn (merge {:consistent true} page)]))

(defn scan [table & [page]]
  (wrap-ex
   (db/scan! (creds) table (merge {:consistent true} page))
   [:scan table (merge {:consistent true} page)]))

(defn scan-with-creds [creds table & [page]]
  (wrap-ex
   (db/scan! creds table (merge {:consistent true} page))
   [:scan-with-creds table (merge {:consistent true} page)]))

(defn empty?? [v] (or (contains? #{[] #{} ""} v) (.isNaN js/Number v)))

(defn clean-empty [val]
  (if (map? val)
    (into {}
          (->> val
               (map #(if (empty?? (second %))
                       [(first %) nil]
                       %))
               (map (fn [[k v]] [k (clean-empty v)]))))
    (if (vector? val)
      (mapv clean-empty (remove empty?? val))
      (if (set? val)
        (set (map clean-empty (remove empty?? val)))
        val))))

(defn put-item [table val]
  (let [val (clean-empty val)]
    (wrap-ex
     (db/put-item! (creds) table val)
     [:put-item table val])))

(defn get-item [table key & [page]]
  (wrap-ex
   (db/get-item! (creds) table key page)
   [:get-item table key page]))

(defn batch-get-item [table keys & [project page]]
  (wrap-ex
   (db/batch-get-item!
    (creds) {table {:consistent true
                    :project project
                    :keys keys}} page)
   [:batch-get-item {table {:consistent true
                            :project project
                            :keys keys}}]))

(defn batch-write-item [updates]
  (let [updates (clean-empty updates)]
    (wrap-ex
     (db/batch-write-item! (creds) updates)
     [:batch-write-item updates])))

(defn update-item [table id val]
  (let [val (clean-empty val)]
    (wrap-ex
     (db/update-item!
      (creds) table id
      (into {} (for [[k v] val]
                 [k (if v [:set v] [:remove])])) {:return :all-new})
     [:update-item table id
      (into {} (for [[k v] val]
                 [k (if v [:set v] [:remove])])) {:return :all-new}])))

;; todo why remove nils?
(defn raw-update-item [table id val]
  (let [val (into {} (remove (comp nil? second) val))]
    (wrap-ex
     (db/update-item! (creds) table id val {:return :all-new})
     [:raw-update-item table id val {:return :all-new}])))

(defn delete-item [table key]
  (wrap-ex
   (db/delete-item! (creds) table key)
   [:delete-item table key]))

(defn get-all [table keys]
  (go-catching
    (loop [[keys & next] (partition-all 100 keys)
           r []]
      (if-not keys
        r
        (recur
         next
         (into r (table (<! (batch-get-item table keys)))))))))
