(ns api.aws
  (:require [cljs.nodejs :as nodejs]
            [promesa.impl] ;; force to avoid conflict with Promise
            [cljs.core.async :as async :refer [<! chan]]
            [api.util :as util])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(def ^:dynamic *parent* nil)
(def ^:dynamic *levels* 0)

(def context-utils (nodejs/require "aws-xray-sdk-core/lib/context_utils.js"))
(def utils (nodejs/require "aws-xray-sdk-core/lib/utils.js"))

(def AWS
  (if (util/local?)
    (nodejs/require "aws-sdk")
    (do
      (def AWSXRay (nodejs/require "aws-xray-sdk-core"))
      (.captureAWS AWSXRay (nodejs/require "aws-sdk")))))

(defn parent [] (when-not (util/local?) (or *parent* (.getSegment AWSXRay))))

(defn initialize-local-aws [profile region]
  (set! (.-region (.-config AWS)) region)
  (set! (.-credentials (.-config AWS))
        (let [screds (def shared-creds (aget AWS "SharedIniFileCredentials"))]
          (new screds #js {:profile profile}))))

(defn capture-http [cli]
  (if (util/local?)
    cli
    (.captureHTTPs AWSXRay cli)))

(defn add-annotation [segment k v]
  (when (and (not (util/local?)) segment v)
    (.addAnnotation segment (clj->js k) (clj->js v))))

(defn add-metadata
  ([segment n v] (add-metadata segment n v nil))
  ([segment n v ns]
   (when (and (not (util/local?)) n v segment)
     (.addMetadata segment (clj->js n) (clj->js v) (clj->js ns)))))

(defn capture [name subsegments]
  (when-not (util/local?)
    (.captureFunc AWSXRay name
                  (fn [subsegment]
                    (doseq [[k v] subsegments]
                      (add-annotation subsegment k v))))))

(defn capture-async [name work]
  (let [ch (chan 1)]
    (if (util/local?)
      (go
        (let [w (work nil)
              r (<! w)]
          (if r
            (async/put! ch r)
            (async/close! ch))))
      (.captureAsyncFunc
       AWSXRay name
       (fn [subsegment]
         (go
           (if-let [r (<! (work subsegment))]
             (async/put! ch r)
             (async/close! ch))))))
    ch))

(defn add-error
  ([er remote?] (when-not (util/local?) (add-error (parent) er remote?)))
  ([subsegment er remote?]
   (when-not (util/local?) (.addError subsegment er remote?))))

(defn close [subsegment]
  (when-not (util/local?) (.close subsegment)))

(defn capture-annotations [subsegments]
  (capture "annotations" subsegments))

(defn add-attribute [subsegment n v]
  (when-not (util/local?)
    (.addAttribute subsegment n v)))

(defn add-operation [subsegment op]
  (when (and subsegment (not (util/local?)))
    (add-annotation subsegment "operation" op)))

(defn capture-node
  ([namespace callback] (capture-node namespace nil callback))
  ([namespace name callback]
   (if (util/local?)
     (callback nil)
     (let [ch (chan 1)]
       (let [subsegment (.addNewSubsegment (parent) namespace)]
         (aset subsegment "namespace" "remote")
         (let [session (.getNamespace context-utils)]
           (.run session
                 (fn []
                   (.setSegment context-utils subsegment)
                   (go
                     (binding [*parent* nil #_(if (= 15 *levels*) *parent* subsegment)
                               ;;*levels* (if (= 15 *levels*) *levels* (inc *levels*))
                               ]
                       (.addRemoteRequestData
                        subsegment
                        (let [r (clj->js {:agent {:protocol "node"}
                                          :path (if name (str ":" name) "")
                                          :method ""})]
                          (aset r "getHeader" (fn [h] namespace))
                          r)
                        (clj->js {:statusCode 200}) false)
                       (if-let [r (<! (callback subsegment))]
                         (async/put! ch r)
                         (async/close! ch)))))))
         ch)))))
