(ns api.s3
  (:require [api.util :as util]
            [api.http :as http]
            [cemerick.url :refer [url url-decode url-encode]]
            [clojure.string :as string]
            [api.crypto :as crypto]
            [api.aws :refer [AWS]]
            [cljs.nodejs :as nodejs]
            [cljs.core.async :as async :refer [chan <!]])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(def fs (nodejs/require "fs"))

(def S3 (aget AWS "S3"))

(defn write-file [file content]
  (.writeFile fs file content
              (fn [err] (println (or err "OK")))))

(defn base64-format [base64]
  (second (re-find #"data:(.*);" base64)))

(defn strip-accents [^String s]
  (let [s (string/lower-case s)
        from "+ąàáäâãåăćčĉęèéëêĝĥìíïîĵľńňòóöőôõðśșšŝťțŭùúüűûñÿýçżźž"
        to "_aaaaaaaaccceeeeeghiiiijlnnooooooossssttuuuuuunyyczzz"]
    (apply str (map (fn [c] (if-let [i (string/index-of from c)] (.charAt to i) c)) s))))

(defn upload [bucket folder key base64]
  (let [key (strip-accents key)
        key (str (.getTime (js/Date.)) "-" key)
        key (str folder (if folder "/") key)
        ch (chan 1)
        b (new S3 #js {})
        data (subs base64 (inc (.indexOf base64 ",")))
        req #js {:Key key
                 :Bucket bucket
                 :ACL "public-read"
                 :CacheControl "max-age=172800"
                 :Body (js/Buffer. data "base64")
                 :ContentEncoding "base64"
                 :ContentType (base64-format base64)}]
    (.putObject b req (fn [err data]
                        (async/put! ch
                                    (if err
                                      err
                                      (str "https://" bucket "/" key)))))
    ch))

(defn upload-raw-base64 [bucket folder key data]
  (let [key (strip-accents key)
        key (str (.getTime (js/Date.)) "-" key)
        key (str folder (if folder "/") key)
        ch (chan 1)
        b (new S3 #js {})
        req #js {:Key key
                 :Bucket bucket
                 :ACL "public-read"
                 :CacheControl "max-age=172800"
                 :Body (js/Buffer. data "base64")
                 :ContentEncoding "base64"
                 :ContentType (cond
                                (.endsWith key ".png")
                                "image/png"
                                (.endsWith key ".jpg")
                                "image/jpeg"
                                (.endsWith key ".ico")
                                "image/x-icon"
                                :else nil)}]
    (.putObject b req (fn [err data]
                        (async/put! ch
                                    (if err
                                      err
                                      (str "https://" bucket "/" key)))))
    ch))

(defn upload-string
  ([bucket folder key data] (upload-string bucket folder key data false true))
  ([bucket folder key data zipped? public?]
   (let [ch (chan 1)]
     (let [key (strip-accents key)
           body (if zipped? data (crypto/zip data))
           key (str (.getTime (js/Date.)) "-" key)
           key (str folder (if folder "/") key)

           b (new S3 #js {})
           req (clj->js {:Key key
                         :Bucket bucket
                         :ACL (if public? "public-read" "private")
                         :CacheControl "max-age=172800"
                         :Body (js/Buffer. body "base64")
                         :ContentEncoding "gzip"
                         :ContentType "text/plain"})]
       (.putObject b req (fn [err data]
                           (async/put! ch
                                       (if err
                                         err
                                         (str "https://" bucket "/" key))))))
     ch)))

(defn get-object [bucket file]
  (let [b (new S3 #js {})
        ch (async/chan 1)]
    (.getObject b #js {:Bucket bucket :Key file}
                (fn [err data]
                  (async/put! ch (or err (aget data "Body")))))
    ch))

(defn upload-binary [bucket folder key contents]
  (let [key (strip-accents key)
        key (str (.getTime (js/Date.)) "-" key)
        key (str folder (if folder "/") key)
        ch (chan 1)
        b (new S3 #js {})
        req #js {:Key key
                 :Bucket bucket
                 :ACL "public-read"
                 :CacheControl "max-age=172800"
                 :Body contents
                 :ContentEncoding "binary"
                 :ContentType (cond
                                (.endsWith key ".png")
                                "image/png"
                                (.endsWith key ".jpg")
                                "image/jpeg"
                                (.endsWith key ".ico")
                                "image/x-icon"
                                :else nil)}]
    (.putObject b req (fn [err data]
                        (async/put! ch
                                    (if err
                                      err
                                      (str "https://" bucket "/" key)))))
    ch))

(defn file-key [url]
  (let [p (aget (.parse (nodejs/require "url") (url-decode url)) "pathname")
        i (.lastIndexOf p "/")]
    (if (= -1 i)
      p
      (subs p (inc i)))))

;; move out?
(defn upload-url [url]
  (go
    (if (.startsWith url (str "https://dynamic." @util/domain))
      url
      (let [file (.readFileSync fs (async/<! (http/url->temp url)))
            base64 (.toString (js/Buffer. file) "base64")
            k (file-key url)]
        (<! (upload-raw-base64 (str "dynamic." @util/domain) "upload" k base64))))))

(defn temporary-url [bucket folder key content-type data expires]
  (let [ch (async/chan 1)]
    (go (let [key (strip-accents key)
              body (crypto/zip-raw data)
              key (str (.getTime (js/Date.)) "-" key)
              key (str folder (if folder "/") key)
              b (new S3 #js {})
              req (clj->js {:Key key
                            :Bucket bucket
                            :ACL "private"
                            :CacheControl "max-age=172800"
                            :Body body
                            :ContentEncoding "gzip"
                            :ContentType content-type})]
          (.putObject
           b req
           (fn [err data]
             (async/put!
              ch
              (.getSignedUrl b "getObject"
                             (clj->js {:Bucket bucket
                                       :Key key
                                       :Expires expires})))))))
    ch))

(defn signed-url [bucket key expires]
  (let [s3  (new S3 #js {})]
    (.getSignedUrl s3 "getObject" (clj->js {:Bucket bucket :Key key :Expires expires}))))
