(ns api.http
  (:require [cljs.core.async :as async]
            [api.aws :refer [capture-http]]
            [cljs.nodejs :as nodejs]
            [api.crypto :as crypto])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(def fs (nodejs/require "fs"))
(def https-cli (capture-http (nodejs/require "https")))
(def http-cli (capture-http (nodejs/require "http")))
(def request (nodejs/require "request"))

(defn get-cli [url]
  (if (= (aget url "protocol") "https:") https-cli http-cli))

(defn GET
  ([url success failure] (GET url {} success failure))
  ([url {:keys [headers]} success failure]
   (let [url (.parse (nodejs/require "url") url)
         _ (when headers (aset url "headers" (clj->js headers)))
         req (.get (get-cli url) url
                       (fn [res]
                         (let [r (atom "")]
                           (.on res "end" (fn [] (success @r)))
                           (.on res "data"
                                (fn [chunk]
                                  (swap! r #(str % chunk)))))))]
     (.setTimeout req 10000 (fn [e] (failure (js/Error. "Timeout"))))
     (.on req "error" (fn [e] (failure e))))))

(defn POST [url post-data]
  (let [ch (async/chan 1)
        url (.parse (nodejs/require "url") url)
        _ (aset url "method" "POST")
        _ (aset url "headers"
                #js {:Content-Type "application/x-www-form-urlencoded"
                     :Content-Length (.byteLength js/Buffer post-data)})
        req (.request (get-cli url) url
             (fn [res]
               (let [r (atom "")]
                 (.on res "end"
                      (fn [] (async/put! ch @r)))
                 (.on res "data"
                      (fn [chunk]
                        (swap! r #(str % chunk)))))))]
    (.on req "error" (fn [e] (async/put! ch e)))
    (.write req post-data)
    (.end req)
    ch))

(defn POST* [url {:keys [headers]} post-data]
  (let [ch (async/chan 1)
        url (.parse (nodejs/require "url") url)
        _ (aset url "method" "POST")
        _ (aset url "headers"
                (clj->js
                 (merge {:Content-Type "application/x-www-form-urlencoded"
                         :Content-Length (.byteLength js/Buffer post-data)}
                        (or headers {}))))
        _ (.log js/console url)
        req (.request (get-cli url) url
             (fn [res]
               (let [r (atom "")]
                 (.on res "end"
                      (fn [] (async/put! ch @r)))
                 (.on res "data"
                      (fn [chunk]
                        (swap! r #(str % chunk)))))))]
    (.on req "error" (fn [e] (async/put! ch e)))
    (.write req post-data)
    (.end req)
    ch))

(defn GET-PLAIN [url]
  (let [ch (async/chan 1)]
    (GET url
         #(async/put! ch {:status :ok :data %})
         #(async/put! ch {:status :er :error (aget % "message")}))
    ch))

(defn GET-JSON [url]
  (let [ch (async/chan 1)]
    (GET url
         #(async/put! ch {:status :ok :data (.parse js/JSON %)})
         #(async/put! ch {:status :er :error (aget % "message")}))
    ch))

(defn GET-EDN
  ([url] (GET-EDN url {}))
  ([url options]
   (let [ch (async/chan 1)]
     (GET url options
          #(async/put!
            ch
            {:status :ok
             :data (when-not (empty? %)
                     (try (js->clj (.parse js/JSON %) :keywordize-keys true)
                          (catch js/Error e
                            (println (pr-str %))
                            (.log js/console e)
                            nil)))})
          #(async/put! ch {:status :er :error (aget % "message")}))
     ch)))

(defn GET-EDN-DATA [url]
  (let [ch (async/chan 1)]
    (go
      (let [{:keys [data error]} (<! (GET-EDN url))]
        (async/put! ch (or data error))))
    ch))

(defn url->buffer [url]
  (let [ch (async/chan 1)]
    (.call (aget request "defaults") request #js {:encoding nil})
    (go ((aget request "get")
         #js {:encoding nil
              :url url} (fn [err res body] (async/put! ch (or err body)))))
    ch))

(defn url->unzip [url]
  (go (crypto/unzip-buffer (<! (url->buffer url)))))

(defn url->temp [url]
  (let [purl (.parse (nodejs/require "url") url)
        ch (async/chan 1)
        i (.lastIndexOf url "/")
        ff (str "/tmp/" (.getTime (js/Date.)) "-" (if (= -1 i) url (subs url (inc i))))
        file (.createWriteStream fs ff)]
    (.get (if (= (aget purl "protocol") "https:")
             (nodejs/require "https")
             (nodejs/require "http"))
          url (fn [response]
                  (-> (.pipe response file)
                      (.on "close" (fn []
                                     (async/put! ch ff))))))
    ch))
