(ns api.crypto
  (:require [cljs.nodejs :as nodejs]
            [api.deasync :refer [loop-while]]
            [cljs.reader :as reader]))

(def node-rsa (nodejs/require "node-rsa"))
(def crypto (nodejs/require "crypto"))
(def zlib (nodejs/require "zlib"))

(defn zip [s]
  (let [r (atom nil)]
    (.gzip zlib s (fn [err buffer] (reset! r (.toString buffer "base64"))))
    (loop-while #(nil? @r))
    @r))

(defn zip-raw [s]
  (let [r (atom nil)]
    (.gzip zlib s (fn [err buffer] (reset! r buffer)))
    (loop-while #(nil? @r))
    @r))

(defn unzip-buffer [b]
  (let [r (atom nil)]
    (.unzip zlib b (fn [err buffer]
                     (if err
                       (do
                         (reset! r "")
                         (throw err))
                       (reset! r (.toString buffer)))))
    (loop-while #(nil? @r))
    @r))

(defn unzip [s]
  (unzip-buffer (js/Buffer. s "base64")))

(defn gen-keys []
  (let [k (new node-rsa #js {:b 1024})
        k (.generateKeyPair k 1024 65537)]
    {:private (.exportKey k "pkcs1-private")
     :public (.exportKey k "pkcs8-public")}))

(defn random-aes-key []
  (.randomBytes crypto 16))

(def aes-alg "aes-128-cbc")
(defn aes-encrypt [key data]
  (let [data (zip data)
        cipher (.createCipheriv crypto aes-alg key key)]
    (str (.update cipher data "base64" "base64") (.final cipher "base64"))))

(defn aes-decrypt [key data]
  (let [key (js/Buffer. key "utf-8")
        decipher (.createDecipheriv crypto aes-alg key key)
        data (str (.update decipher data "base64" "base64")
                  (.final decipher "base64"))]
    (unzip data)))

(defn encrypt-decrypt [^String key s for-encryption public?]
  (let [s (js/Buffer. s (if for-encryption "utf8" "base64"))
        key #js {:key key
                 :padding 1}]
    (if for-encryption
      (.toString (if public?
                   (.publicEncrypt crypto key s)
                   (.privateEncrypt crypto key s)) "base64")
      (if public?
        (.publicDecrypt crypto key s)
        (.privateDecrypt crypto key s)))))

(defn prdecrypt [key s]
  (encrypt-decrypt key s false false))

(defn prencrypt [key s]
  (encrypt-decrypt key s true false))

(defn decrypt [key s]
  (encrypt-decrypt key s false true))

(defn encrypt [key s]
  (encrypt-decrypt key s true true))

(defn hencrypt [key s]
  (let [aes-key (random-aes-key)]
    (pr-str [(encrypt key aes-key) (aes-encrypt aes-key (pr-str s))])))

(defn hdecrypt [key t]
  (let [[aes-key s] (reader/read-string t)]
    (reader/read-string (aes-decrypt (decrypt key aes-key) s))))

(defn prhencrypt [key s]
  (let [aes-key (random-aes-key)]
    (pr-str [(prencrypt key aes-key)
             (aes-encrypt aes-key (pr-str s))])))

(defn prhdecrypt [key t]
  (let [[aes-key s] (reader/read-string t)]
    (reader/read-string (aes-decrypt (prdecrypt key aes-key) s))))
