(ns api.charts)

(def colors-fixed
  ["#4D4D4D" "#5DA5DA" "#FAA43A" "#60BD68" "#F17CB0" "#B2912F" "#B276B2"
   "#DECF3F" "#F15854"
   "#E6E6FA" "#D8BFD8" "#DDA0DD" "#EE82EE" "#DA70D6" "#FF00FF" "#FF00FF"
   "#BA55D3" "#9370DB" "#8A2BE2" "#9400D3" "#9932CC" "#8B008B" "#800080"
   "#4B0082" "#483D8B" "#6A5ACD" "#7B68EE" "#FFC0CB" "#FFB6C1" "#FF69B4"
   "#FF1493" "#DB7093" "#C71585" "#FFA07A" "#FA8072" "#E9967A" "#F08080"
   "#CD5C5C" "#DC143C" "#B22222" "#8B0000" "#FF0000" "#FF4500" "#FF6347"
   "#FF7F50" "#FF8C00" "#FFA500" "#FFFF00" "#FFFFE0" "#FFFACD" "#FAFAD2"
   "#FFEFD5" "#FFE4B5" "#FFDAB9" "#EEE8AA" "#F0E68C" "#BDB76B" "#FFD700"
   "#FFF8DC" "#FFEBCD" "#FFE4C4" "#FFDEAD" "#F5DEB3" "#DEB887" "#D2B48C"
   "#BC8F8F" "#F4A460" "#DAA520" "#B8860B" "#CD853F" "#D2691E" "#8B4513"
   "#A0522D" "#A52A2A" "#800000" "#556B2F" "#808000" "#6B8E23" "#9ACD32"
   "#32CD32" "#00FF00" "#7CFC00" "#7FFF00" "#ADFF2F" "#00FF7F" "#00FA9A"
   "#90EE90" "#98FB98" "#8FBC8F" "#3CB371" "#2E8B57" "#228B22" "#008000"
   "#006400" "#66CDAA" "#00FFFF" "#00FFFF" "#E0FFFF" "#AFEEEE" "#7FFFD4"
   "#40E0D0" "#48D1CC" "#00CED1" "#20B2AA" "#5F9EA0" "#008B8B" "#008080"
   "#B0C4DE" "#B0E0E6" "#ADD8E6" "#87CEEB" "#87CEFA" "#00BFFF" "#1E90FF"
   "#6495ED" "#4682B4" "#4169E1" "#0000FF" "#0000CD" "#00008B" "#000080"
   "#191970" "#FFFFFF" "#FFFAFA" "#F0FFF0" "#F5FFFA" "#F0FFFF" "#F0F8FF"
   "#F8F8FF" "#F5F5F5" "#FFF5EE" "#F5F5DC" "#FDF5E6" "#FFFAF0" "#FFFFF0"
   "#FAEBD7" "#FAF0E6" "#FFF0F5" "#FFE4E1" "#DCDCDC" "#D3D3D3" "#C0C0C0"
   "#A9A9A9" "#808080" "#696969" "#778899" "#708090" "#2F4F4F"])

(defn data-percentages [data]
  (let [total (apply + 0 (map :value (:sectors data)))]
    (assoc data :sectors
           (mapv #(assoc %
                         :percentage (/ (:value %) total)
                         :color %2) (:sectors data) (cycle colors-fixed)))))

(defn calculate-sectors [{:keys [sectors size]}]
  (loop [[{:keys [percentage label color] :as sector} & data] sectors
         sectors []
         l (/ size 2) a 0 a-rad 0 z 0 x 0 y 0 xx 0 yy 0 rr 0]
    (if-not sector
      sectors
      (let [a (* 360 percentage)
            a-calc (if (> a 180)
                     (do
                       (println a (- 360 a))
                       (- 360 a))
                     a)
            a-rad (/ (* a-calc js/Math.PI) 180)
            z (js/Math.sqrt (- (* 2 l l) (* 2 l l (js/Math.cos a-rad))))
            x (* l
                 (if (<= a-calc 90)
                   (.sin js/Math a-rad)
                   (.sin js/Math (* (- 180 a-calc) (/ js/Math.PI 180)))))
            y (js/Math.sqrt (- (* z z) (* x x)))
            yy y
            [xx arc-sweep] (if (<= a 180)
                             [(+ l x) 0]
                             [(- l x) 1])]
        (recur data
               (conj sectors
                     {:percentage percentage
                      :label label
                      :arc-sweep arc-sweep
                      :color color
                      :ll l
                      :xx xx
                      :yy yy
                      :rr rr}) l a a-rad z x y xx yy (+ rr a))))))

(defn radians [x]
  (* (/ js/Math.PI 180.0) (double x)))

(defn pie [data]
  (let [data (data-percentages data)
        sectors (calculate-sectors data)]
    [:div
     [:svg {:style {:width (:size data)
                    :height (:size data)}}
      (doall
       (for [n (range (count sectors))]
         (let [{:keys [percentage label color ll xx yy rr arc-sweep]} (nth sectors n)]
           ^{:key n}
           [:g
            [:path {:fill color
                    :transform (str "rotate(" rr "," ll "," ll ")")
                    :d (str "M" ll "," ll " L" ll ",0 A" ll "," ll " 0 " arc-sweep ",1 " xx ", " yy " z")}]
            [:path {:stroke "#000000"
                    :stroke-opacity "0.2"
                    :fill "transparent"
                    :transform (str "rotate(" rr "," ll "," ll ")")
                    :d (str "M" ll "," ll " L" ll ",0 A" ll "," ll " 0 " arc-sweep "1 " xx ", " yy " z")}]])))
      (doall
       (for [n (range (count sectors))]
         (let [{:keys [percentage label color ll xx yy rr]} (nth sectors n)
               radius (+ 40 (* (mod n 4) 10))
               rr (radians (+ rr (/ (* percentage 360) 2) -90))
               x (+ ll (* (- ll radius) (js/Math.cos rr)))
               y (+ ll (* (- ll radius) (js/Math.sin rr)))]
           ^{:key (str "label" n)}
           [:text {:x x
                   :y y
                   :style {:text-shadow "0 0 1px #AAA"
                           :transform "translate(-30px, 10px)"
                           :font-weight "bold"}
                   :font-size "10"} label])))]
     [:div
      (doall
       (for [n (range (count sectors))]
         (let [{:keys [color label]} (nth sectors n)]
           ^{:key n}
           [:div {:style {:display "inline-block"
                          :padding 5}}
            [:div {:style {:width 20 :height 20 :background-color color
                           :vertical-align "middle"
                           :margin-right 5
                           :border "1px solid black"
                           :display "inline-block"}}]
            [:span {:style {:vertical-align "middle"}}
             label]])))]]))

(defn line-chart [data]
  (let [{:keys [width height lines labels]} data
        lines (vec (map #(assoc % :color %2) lines (cycle colors-fixed)))
        step (/ width (dec (apply max 0 (map (comp count :values) lines))))
        max (apply max 0 (mapcat :values lines))]
    [:div {:style {:display "inline-block"
                   :padding "0 50"}}
     [:svg {:style {:width width :height height
                    :overflow "visible"
                    ;;:background-color "#111"
                    :border-bottom "1px dotted #DDD"
                    :border-left "1px dotted #DDD"
                    :margin "25px 10px"}}
      (doall
       (for [n (range (count lines))]
         (let [{:keys [label values color]} (nth lines n)]
           ^{:key n}
           [:polyline
            {:fill "none"
             :stroke color
             :stroke-linejoin "round"
             :stroke-width "3"
             :points (apply str
                            (interpose
                             " "
                             (map #(let [% (or % 0)]
                                     (str (* step %) ","
                                          (- height (/ (* (nth values %) height) max))))
                                  (range (count values)))))}])))
      (doall
       (for [n (range (count labels))]
         (let [label (nth labels n)]
           ^{:key n}
           [:text {:x (* step n)
                   :y (+ 10 height)
                   :text-anchor "middle"
                   :style {:text-shadow "0 0 1px #AAA"}
                   :font-size "7"} label])))
      (let [height (- height 10)]
        (doall
         (for [n [0 (/ max 2) max]]
           ^{:key (str "vertical" n)}
           [:text {:x -5
                   :y (+ 5 (- height (/ (* n height) max)))
                   :text-anchor "end"
                   :style {:text-shadow "0 0 1px #AAA"}
                   :font-size "10"} (str n)])))]
     [:div {:style {:text-align "center"}}
      (doall
       (for [n (range (count lines))]
         (let [{:keys [color label]} (nth lines n)]
           ^{:key (str "label" n)}
           [:div {:style {:display "inline-block"
                          :padding 5}}
            [:div {:style {:width 20 :height 20 :background-color color
                           :vertical-align "middle"
                           :margin-right 5
                           :border "1px solid black"
                           :display "inline-block"}}]
            [:span {:style {:vertical-align "middle"}}
             label]])))]]))
