(ns api.remote
  (:require
   [api.util :as util]
   [api.cookie :as cookie]
   [goog.net.XhrIo :as xhr]
   [re-frame.db :refer [app-db]]
   [glossop.core :refer-macros [<? go-catching]]
   [reagent.core :as reagent]
   [cognitect.transit :as t]
   [re-frame.core :refer [reg-event-db path reg-sub dispatch dispatch-sync subscribe]]
   [cljs.core.async :as async :refer [chan close! put! <! timeout]])
  (:require-macros [cljs.core.async.macros :refer [go alt!]]))

(def endpoint (atom nil))

(defn GET [url]
  (let [ch (chan 1)]
    (xhr/send url
              (fn [event]
                (let [res (-> event .-target .getResponseText)]
                  (go (>! ch res)
                      (close! ch)))))
    ch))

(defn cordova-post [url content]
  (let [ch (chan 1)]
    (.post js/cordovaHTTP url
           (clj->js {:body content})
           #js {"Accept-Encoding" "gzip"
                "Cookie" (if-let [token (:state/token @app-db)]
                           (str "token=" token)
                           "")
                "Content-Type" "application/x-www-form-urlencoded"}
           (fn [response]
             (let [res (aget response "data")
                   status (aget response "status")]
               (async/put! ch (if (= 200 status)
                                {:status :ok :body res}
                                {:status :er :body res}))
               (close! ch)))
           (fn [response]
             (async/put! ch {:status :er :body (aget response "error")})))
    ch))

(defn post [url content]
  (let [content (.encodeURIComponent js/window content)
        ch (chan 1)
        ctoken (cookie/get "token")
        token (or ctoken (:state/token @app-db))]
    (when (and (not ctoken) token)
      (cookie/set "token" token))
    (xhr/send url
              (fn [event]
                (let [res (-> event .-target .getResponseText)
                      status (.getStatus (.-target event))]
                  (go (>! ch (if (= 200 status)
                               {:status :ok :body res}
                               {:status :er :body res}))
                      (close! ch))))
              "POST" content #js {"Access-Control-Allow-Origin" "*"})
    ch))

(defn api-background
  ([path] (api-background path nil))
  ([path & params]
   (go-catching
    (let [{:keys [status body] :as all}
          (<! ((if (and (exists? js/cordovaHTTP)
                        (exists? js/device)
                        #_(not util2/android?)) cordova-post post)
                  (if (exists? js/cordovaHTTP)
                    (str @endpoint path)
                    path)
                (t/write (t/writer :json) (vec params))))]
       (let [body (try
                    (t/read (t/reader :json) body)
                    (catch js/Error e
                      (.log js/console e)
                      (dispatch [:api-error {:msg "ERROR CALLING SERVER"}])
                      (throw e)))]
         (if (= status :ok)
           body
           (do
             (println "API ERROR ON" path params)
             (dispatch [:api-error (if (string? body)
                                       {:msg body}
                                       body)])
               (if (and (= "/send-message" path) (string? body))
                 {:status :er :msg body :details :unknown}
                 body)
               ;; REMOVE?
               #_(throw (js/Error. (:msg body))))))))))

(defn api
  ([path] (api path nil))
  ([path & params]
   (dispatch [:loading])
   ;; DO NOT USE go-catching <?
   (go
     (let [r (<! (apply api-background path params))]
       (dispatch [:unloading])
       r))))
