(ns api.forms
  (:require cljsjs.react
            cljsjs.react-input-mask
            [clojure.string :refer [lower-case]]
            [cljs-time.core :refer [date-time]]
            [cljs.reader :as reader]
            [api.util :as util]
            [api.messages :as messages]
            [api.remote :as remote]
            [re-frame.core :refer [reg-event-db path reg-sub dispatch-sync subscribe]]
            [cljs.core.async :as async :refer [chan close! put! <! timeout]]
            [cljs-time.format :as time]
            [reagent.core :as reagent]
            [goog.string :as gstring]
            [goog.string.format])
  (:require-macros [cljs.core.async.macros :refer [go go-loop alt!]]))

(def masked (reagent/adapt-react-class js/InputElement))

(defn checked [e]
  (.-checked (.-target e)))

(defn control [label & form]
  (into [:div.form-group
         [:label (let [l (if (keyword? label) (messages/with label) label)]
                   (if (empty? l) (gstring/unescapeEntities "&nbsp;") l))]]
        form))

(defn control-with [label class & form]
  (into [:div.form-group {:class class}
         [:label (if (keyword? label) (messages/with label) label)]]
        form))

(def ^:dynamic *form-data*)
(defn field
  ([id editor] (field id editor {}))
  ([id editor attrs & body]
   [control id
    (into
     [editor
      (merge
       {:dispatch (:dispatch *form-data*)
        :val (id (:data *form-data*))
        :id id}
       attrs)]
     (vec body))]))

(defn format-date [d f]
  (time/unparse (time/formatter f) (date-time d)))

(defn render-date [date]
  (format-date date "dd/mm/yyyy"))

(defn render-datetime [date]
  (time/unparse (time/formatter "dd/MM/yyyy HH:mm:ss") (date-time date)))

(defn parse-double [v]
  (let [v (when v (clojure.string/replace v #"[,]" "."))]
    (if (number? v)
      v
      (if (empty? v) nil (let [n (js/parseFloat v)] (if (js/isNaN n) nil n))))))

(defn parse-int [v]
  (let [v (apply str (remove #(= \, %) v))]
    (if (number? v)
      v
      (if (empty? v) nil (let [n (js/parseInt v)] (if (js/isNaN n) nil n))))))

(defn format-double [d]
  (if d
    (let [n (neg? d)]
      (str (if n "-" "") (gstring/format "%.2f" (.abs js/Math d))))
    ""))

(defn value-of [f] (.-value (.-target f)))

(defn vec-remove [coll pos]
  (vec (concat (subvec coll 0 pos) (subvec coll (inc pos)))))

(defn vec-insert [coll pos item]
  (vec (concat (subvec coll 0 pos) [item] (subvec coll pos))))

(defn filter-data [data search]
  (vec
   (remove
    #(= -1
        (.indexOf
         (lower-case (:entity/str %))
         (lower-case search))) data)))

(defn text-input [{:keys [dispatch id val type on-change args] :as attrs}]
  ^{:key id}
  [:input.form-control
   (merge
    {:type (or type "text")
     :value val
     :onChange #(do (dispatch-sync (conj (into [dispatch] (or args []))
                                         {id (value-of %)}))
                    (when on-change (on-change)))}
    (dissoc attrs :dispatch :id :val :on-change :path :args))])

(defn color-input [{:keys [dispatch id val type on-change args] :as attrs}]
  ^{:key id}
  [:input.form-control
   (merge
    {:type "color"
     :style {:width 60
             :height 50}
     :value val
     :onChange #(do (dispatch-sync (conj (into [dispatch] (or args []))
                                         {id (value-of %)}))
                    (when on-change (on-change)))}
    (dissoc attrs :dispatch :id :val :on-change :path :args))])



(defn textarea-input [{:keys [dispatch id val type] :as attrs}]
  ^{:key id}
  [:textarea.form-control
   (merge
    {:type (or type "text")
     :value val
     :onChange #(dispatch-sync [dispatch {id (value-of %)}])}
    (dissoc attrs :dispatch :id :val))])

(defn file-input [{:keys [id doc multiple on-change] :as attrs}]
  [:div.flex.flex1.flex-column
   [:div [:b (if multiple
               (apply str (interpose ", " (map :name (id @doc))))
               (:name (id @doc)))]]
   [:input
    (merge
     {:type "file"
      :multiple (if multiple "multiple" "")
      :on-change
      (fn [e]
        (let [target (.-target e)
              files (.-files target)
              a (atom [])]
          (doseq [n (range (alength files))]
            (let [file (aget files n)
                  reader (js/FileReader.)]
              (aset reader "onerror" #(swap! a conj nil))
              (aset reader "onabort" #(swap! a conj nil))
              (aset reader "onloadend"
                    #(let [b64 (.-result (.-target %))
                           b64 (.substring b64 (inc (.indexOf b64 ",")))]
                       (swap! a conj {:name (.-name file) :data b64})
                       (when (= (count @a) (alength files))
                         (set! (.-value target) nil)
                         (swap! doc assoc id
                                (if multiple
                                  (vec (filter identity @a))
                                  (first @a)))
                         (when on-change
                           (on-change (id @doc))))))
              (.readAsDataURL reader file)))))}
     (dissoc attrs :on-change :multiple :id :doc))]])

(defn checkbox-input [{:keys [dispatch id val] :as attrs}]
  [:input
   (merge
    {:type "checkbox"
     :checked (true? val)
     :onChange #(dispatch-sync [dispatch {id (.-checked (.-target %))}])}
    (dissoc attrs :dispatch :id :val))])

(defn date-vector->map [[y m d]]
  {:year y
   :month (inc m)
   :day d})

(defn nan? [c]
  (js/isNaN (js/parseInt c)))

(defn has-nan? [v]
  (some true? (map nan? v)))

(defn has-nan-comma? [v]
  (some true? (map #(or (#{"-" "."} %) (nan? %)) v)))

(defn double-input [_]
  (let [temp (reagent/atom nil)
        blur (fn [dispatch id tipping?]
               (when @temp
                 (let [v (parse-double @temp)]
                   (dispatch-sync [dispatch {id v}])
                   (when-not tipping? (reset! temp nil)))))]
    (fn [{:keys [dispatch id val formatter] :as attrs}]
      [:input.form-control
       (merge
        {:type :text
         :on-blur #(blur dispatch id false)
         :value (or @temp ((or formatter format-double) val))
         :on-change #(let [v (value-of %)]
                       (when-not (has-nan-comma? v)
                         (reset! temp v)
                         (try (blur dispatch id true) (catch js/Error e nil))))}
        (dissoc attrs :dispatch :id :val :formatter))])))

(defn int-input [{:keys [dispatch id val on-change] :as attrs}]
  [:input.form-control
   (merge
    {:type :text
     :value (str val)
     :on-change
     #(let [v (value-of %)]
        (when-not (has-nan? v)
          (if on-change
            (on-change (parse-int v))
            (dispatch-sync [dispatch {id (parse-int v)}]))))}
    (dissoc attrs :dispatch :id :val :on-change))])

(defn select-list [{:keys [id source doc] :as attrs}]
  [:div {:style {:display "inline-block"
                 :vertical-align "top"}}
   (let [val (set (map #(if (map? %) (:db/id %) %) (get @doc id)))
         l (vec (source doc))]
     (doall
      (for [n (range (count l))]
        (let [i (nth l n)]
          ^{:key n}
          [:label.checkbox {:style {:display "block"}}
           (let [c (get val (:db/id i))]
             [:input
              {:type "checkbox"
               :checked c
               :on-change #(swap! doc assoc id
                                 (vec ((if c disj conj) val (:db/id i))))}])
           (:entity/str i)]))))])

(defn linked-list [{:keys [id source doc] :as attrs}]
  (let [v (get @doc id)
        items (source doc)]
    (when (and (nil? v) (not (empty? items)))
      (swap! doc assoc id (first items)))
    [:select (merge
              {:value (if (map? v) (:db/id v) v)
               :on-change #(swap! doc assoc id (parse-int (value-of %)))}
              (dissoc attrs :id :source :doc))
     (doall
      (for [i items]
        ^{:key (:db/id i)}
        [:option {:value (:db/id i)} (:entity/str i)]))]))

(defn list-editor
  ([s editor] (list-editor s editor editor))
  ([s view editor] (list-editor s view editor #(not (empty? %))))
  ([s view editor validate]
   (let [nn (reagent/atom {})
         cur (memoize (fn [n] (reagent/cursor s [n])))]
     (fn []
       (let [items (or @s [])]
         [:table.table.table-bordered.table-striped
          {:style {:width "auto" :margin-bottom "0"
                   :display "inline-block"
                   :vertical-align "text-top"}}
          [:tbody
           (doall
            (for [n (range (count items))]
              ^{:key n}
              [:tr
               [:td (view (cur n))]
               [:td [:button.abutton-red
                     {:on-click #(swap! s (fn [v] (vec-remove v n)))
                      :type "button"}
                     "×"]]]))
           (when editor
             ^{:key "__new"}
             [:tr
              [:td (editor nn)]
              [:td [:button.abutton
                    {:on-mouse-down #(do (.preventDefault %)
                                       (.stopPropagation %))
                     :type "button"
                     :on-click
                     #(let [val @nn]
                        (when (validate val)
                          (swap! s (fn [v] (conj (or v []) val)))
                          (reset! nn nil)))}
                    "+"]]])]])))))

(defn sorted-val [v]
  (cond
    (map? v) (into (sorted-map) v)
    (set? v) (into (sorted-set) v)
    :else v))

(defn select-input [{:keys [dispatch id val on-change] :as attrs} & options]
  (let [default (:value (second (first options)))]

    #_(when (and (not val) default)
      (dispatch-sync [dispatch {id default}]))
    (into
     [:select.form-control
      (merge
       {:value (pr-str (sorted-val val))
        :onChange
        #(let [v (value-of %)]
           (dispatch-sync [dispatch {id (reader/read-string v)}])
           (when on-change (on-change)))}
       (dissoc attrs :dispatch :id :val :on-change))]
     (vec (map #(update-in % [1 :value] (comp pr-str sorted-val)) options)))))

(defn hour-input [{:keys [dispatch val k id]}]
  [masked {:mask "99:99"
           :value (or (k val) "")
           :onChange #(dispatch-sync [dispatch {id (assoc val k (value-of %))}])
           :style {:width 50 :text-align "center"}}])

(defn empty-hour? [h]
  (empty? (apply str (remove #(#{"_" ":"} %) h))))

(defn open-hours-input [{:keys [dispatch id val type] :as attrs}]
  (let [has-period2?
        (not (empty?
              (filter (fn [[k v]]
                        (and
                         (not (empty-hour? v))
                         (.indexOf (name k) "_2_"))) val)))]
    ^{:key id}
    [:table.table.table-striped
     {:style {:background-color "white"
              :width (if has-period2? 500 340)}}
     [:thead
      [:tr
       [:th (messages/with :day)]
       [:th (str (messages/with :period) " 1")]
       (when has-period2?
         [:th (str (messages/with :period) " 2")])]]
     [:tbody
      (doall
       (for [d [:mon :tue :wed :thu :fri :sat :sun]]
         (let [o1 (keyword (str (name d) "_1_open"))
               c1 (keyword (str (name d) "_1_close"))
               partial-period1?
               (or
                (not (empty-hour? (o1 val)))
                (not (empty-hour? (c1 val))))
               has-period1? (and
                             (not (empty-hour? (o1 val)))
                             (not (empty-hour? (c1 val))))]
           ^{:key d}
           [:tr
            [:td {:style
                  (when-not partial-period1?
                    {:color "red"
                     :text-decoration "line-through"})}
             (messages/with d)]
            [:td
             [hour-input {:val val :dispatch dispatch :k o1 :id id}]
             [hour-input {:val val :dispatch dispatch :k c1 :id id}]]
            (when has-period2?
              (if has-period1?
                [:td
                 [hour-input {:val val :dispatch dispatch
                              :k (keyword (str (name d) "_2_open"))
                              :id id}]
                 [hour-input {:val val :dispatch dispatch
                              :k (keyword (str (name d) "_2_close"))
                              :id id}]]
                [:td]))])))]]))

(defn input-upload [_]
  ;; TODO
  (let [loading (reagent/atom false)]
    (fn [{:keys [dispatch id val on-change]}]
      [:div
       [:input
        {:type "file"
         :onClick #(.stopPropagation %)
         :onChange
         (fn [e]
           (let [reader (js/FileReader.)
                 target (.-target e)
                 file (aget (.-files target) 0)]
             (aset reader "onerror" #(set! (.-value target) nil))
             (aset reader "onabort" #(set! (.-value target) nil))
             (aset reader "onloadend"
                   #(let [b64 (.-result (.-target %))]
                      (go
                        (reset! loading true)
                        ;; todo error?
                        (let [val (<! (remote/api "/upload"
                                                  {:key (.-name file) :data b64}))]
                          (dispatch-sync [dispatch {id val}])
                          (when on-change (on-change val))
                          (reset! loading false)))
                      (set! (.-value target) nil)))
             (.readAsDataURL reader file)))}]
       (when @loading
         [util/icon-svg {:style {:color "#000000"
                                 :width 30
                                 :height 30
                                 :margin 3}}
          :loading])])))

(defn image-input [{:keys [val] :as attrs}]
  [:div.image-editor
   {:style {:min-width "30px"
            :min-height "30px"}}
   (when val [:a {:href val :target "_blank"}
              [:img {:src val :height 100 :style {:margin 5}}]])
   [input-upload attrs]])

(defn single-file-input [{:keys [val] :as attrs}]
  [:div
   (when val [:a {:href val :target "_blank"} val])
   [input-upload attrs]])

(defn geo->points [geo]
  (reduce into [] (mapv #(:coordinates (:geometry %)) (:features geo))))

(defn points->geo [points]
  {:type "FeatureCollection"
   :features
   (mapv
    (fn [points]
      {:type "Feature"
       :geometry
       {:type "Polygon"
        :coordinates [points]}
       :properties {}}) points)})

(defn load-google-maps []
  (when (exists? js/document)
    (let [script (.createElement js/document "script")]
      (aset script "src" "https://maps.googleapis.com/maps/api/js?v=3.exp&callback=initGoogleMaps&key=AIzaSyA0k7Tts6rbx6ctsIHm1UupzLZIvTWG_4k&libraries=geometry,drawing")
      (.appendChild (aget (.getElementsByTagName js/document "head") 0) script))))

(defn gmap-component-loaded [{:keys [dispatch id]}]
  (let [gmap    (atom nil)
        old (atom nil)
        lat-lng (atom nil)
        delete-all
        (fn []
          (let [data (.-data (:map @gmap))]
            (.forEach data (fn [f] (.remove data f)))))
        update
        (fn [comp]
          (let [{:keys [latitude longitude val]} (reagent/props comp)
                latlng (js/google.maps.LatLng. latitude longitude)]
            (when-not (= @lat-lng [latitude longitude])
              (reset! lat-lng [latitude longitude])
              (.panTo (:map @gmap) latlng))
            (.setPosition (:marker @gmap) latlng)
            (when-not (= @old val)
              (delete-all)
              (.addGeoJson (.-data (:map @gmap)) (clj->js (points->geo val))))
            (reset! old val)))]
    (reagent/create-class
     {:reagent-render
      (fn [attrs]
        [:div (dissoc attrs :dispatch :id :latitude :longitude :val)])

      :component-did-mount
       (fn [comp]
         (let [canvas  (reagent/dom-node comp)
               {:keys [dispatch id latitude longitude val]} (reagent/props comp)
               latlng (js/google.maps.LatLng. latitude longitude)
               gm      (js/google.maps.Map.
                        canvas
                        (clj->js {:zoom 13
                                  :center latlng
                                  :scrollwheel false}))
               marker  (js/google.maps.Marker.
                        (clj->js {:map gm :title "Restaurant"
                                  :position latlng}))
               data (aget gm "data")
               center-map
               #(let [[latitude longitude] @lat-lng]
                  (.panTo gm (js/google.maps.LatLng. latitude longitude)))
               polychange
               (fn []
                 (.toGeoJson
                  data
                  (fn [json]
                    (let [val (geo->points
                               (js->clj json :keywordize-keys true))]
                      (reset! old val)
                      (dispatch-sync [dispatch {id val}])))))]
           ;;(.call (aget data "setControls") data #js ["Polygon"])
           #_(.setStyle data #js {:editable true
                                :draggable true
                                :fillColor "#000"
                                :fillOpacity 0.3
                                :strokeWeight 5
                                :clickeable true})
           #_(let [center (.createElement js/document "img")]
             (aset center "src" (util/icon :target))
             (aset center "className" "google-maps-icon")
             (aset center "index" 1)
             (.addEventListener center "click" center-map)
             (.push (aget (.-controls gm)
                          js/google.maps.ControlPosition.TOP_LEFT) center))

           #_(let [clear (.createElement js/document "img")]
             (aset clear "src" (util/icon :delete))
             (aset clear "className" "google-maps-icon")
             (aset clear "index" 1)
             (.addEventListener clear "click" delete-all)
             (.push (aget (.-controls gm)
                          js/google.maps.ControlPosition.TOP_LEFT) clear))

           (.addListener data "addfeature" polychange)
           (.addListener data "removefeature" polychange)
           (.addListener data "setgeometry" polychange)
           (reset! gmap {:map gm :marker marker})
           (update comp)))

       :component-did-update update
      :display-name "google-maps"})))

(defn gmap-component [attrs]
  (let [google-maps (subscribe [:google-maps-initialized])]
    (when-not @google-maps
      (load-google-maps))
    (fn [attrs]
      (if @google-maps
        [gmap-component-loaded attrs]
        [:div (dissoc attrs :dispatch :id :latitude :longitude :val)
         "Loading google maps..."]))))

(defn polygon-input [{:keys [dispatch id val] :as attrs}]
  ^{:key id}
  [gmap-component (assoc attrs :val val)])

(defn geocoder [{:keys [on-change dispatch] :as attrs}]
  (let [val (reagent/atom nil)
        options (reagent/atom nil)
        google-maps (subscribe [:google-maps-initialized])
        click-outside #(reset! options nil)]
    (util/add-window-listener "mouseup" click-outside)
    (fn [{:keys [type] :as attrs}]
      [:div.dropdown {:class (when @options "open")}
       [:div.input-group {:style {:width 250}}
        [:input.form-control
         (merge
          {:type (or type "text")
           :placeholder (messages/with :search-address)
           :value @val
           :onChange #(do (reset! val (value-of %))
                          (when on-change (on-change)))}
          (dissoc attrs :dispatch :id :on-change))]

        [:div.input-group-btn
         [:button.btn.btn-secondary
          {:onClick
           #(go
              (when-not @google-maps
                (load-google-maps))
              (loop []
                (<! (async/timeout 100))
                (when-not @google-maps
                  (recur)))
              (let [geo (js/google.maps.Geocoder.)]
                  (.geocode
                   geo #js {:address @val}
                   (fn [results status]
                     (when (= status "OK")
                       (reset! options (js->clj results :keywordize-keys true)))))))}
          [util/icon-svg {:style {:color "black"
                                  :display "inline-block"
                                  :width 20}} :search]]]]
       [:div.dropdown-menu
        (doall
         (for [o @options]
           ^{:key (:formatted_address o)}
           [:span.dropdown-item
            {:style {:cursor "pointer"}
             :onClick
             (fn []
               (let [location {:lat (js/parseFloat (.lat (:location (:geometry o))))
                               :lng (js/parseFloat (.lng (:location (:geometry o))))}]
                 (dispatch-sync [dispatch (assoc o :location location)]))
               (reset! options nil))}
            (:formatted_address o)]))]])))


(defn render-iframe [e]
  (let [doc (.-contentDocument (reagent/dom-node e))]
    (if (= "complete" (.-readyState doc))
      (do
        (set! (.-innerHTML (.-body doc))
              (get (aget (aget e "props") "argv") 2)))
      (go (<! (timeout 1))
          (render-iframe e)))))

(defn iframe [attrs e]
  (reagent/create-class
   {:reagent-render (fn [attrs html] [:iframe attrs])
    :component-did-mount render-iframe
    :component-did-update render-iframe}))
