(ns api.util
  (:require clojure.string
            [cljs.core.async :as async]
            [api.env :as env])
  (:require-macros [api.icons :refer [def-icons]]))

(def domain (atom nil))

(defn local? []
  (env/local?))

(def-icons icons)

(defn chan? [c]
  (satisfies? cljs.core.async.impl.protocols/ReadPort c))

(defn add-listener [target event handler]
  (when (exists? target)
    (.addEventListener target event handler false)))

(defn add-window-listener [event handler]
  (when (exists? js/window)
    (add-listener js/window event handler)))

(defn rand-hex [n]
  (apply str (for [n (range n)] (.toString (rand-int 16) 16))))

(defn rand-num [n]
  (apply str (for [n (range n)] (rand-int 10))))

(defn proxy [file]
  (if (local?)
    (str "http://localhost:3000/" file)
    (str (str "proxy?https://static." @domain "/") file)))

(defn img [file]
  (if (local?)
    (str "http://localhost:3000/" file)
    (str "https://static." @domain "/" file)))

(defn icon [k]
  (let [k (name k)]
    (if (local?)
      (str "http://localhost:3000/icons/" k ".svg")
      (str "https://static." @domain "/icons/" k ".svg"))))

(defn icon-svg [attrs k]
  (let [html (icons (str (name k) ".svg"))
        color (:color (:style attrs))
        color (or color "#000000")
        color (when (or (.startsWith color "#")
                        (.startsWith color "rgb"))
                color)
        html (if-not color
               html
               (-> html
                   (clojure.string/replace #"fill=\"#000000\""
                                           (str "fill=\"" color "\""))
                   (clojure.string/replace #"fill=\"rgb(.*)\""
                                           (str "fill=\"" color "\""))))
        html (if-not color
               html
               (clojure.string/replace
                html #"<svg "
                (str "<svg fill=\"" (:color (:style attrs)) "\"")))]
    [:div
     (merge
      {:dangerouslySetInnerHTML
       {:__html html}}
      attrs)]))

(defn append [db list id val]
  (if-let [curr (first (filter #(= (id %) (id val)) (get-in db list)))]
    (update-in db list #(mapv (fn [i] (if (= curr i) val i)) %))
    (update-in db list #(conj (vec (or % [])) val))))

(defn dissoc-in [m [k & ks :as keys]]
  (if ks
    (if-let [nextmap (get m k)]
      (let [newmap (dissoc-in nextmap ks)]
        (if (seq newmap)
          (assoc m k newmap)
          (dissoc m k)))
      m)
    (dissoc m k)))

(defn window-hash []
  (when (exists? js/window)
    window.location.pathname))

(defn document-attr [k]
  (when (exists? js/document)
    (.getAttribute js/document.body k)))

(defn data-uri->blob [data mime]
  (let [array (js/Uint8Array.
               (clj->js (map #(.charCodeAt data %) (range (count data)))))]
    (js/Blob. #js [array] #js {:type mime})))

(defn data-uri->url [data mime]
  (js/URL.createObjectURL (data-uri->blob data mime)))
