;  Copyright (C) 2020 Gabriel Ash

; This program and the accompanying materials are made available under
; the terms of the Eclipse Public License 2.0 which is available at
; http://www.eclipse.org/legal/epl-2.0 .

; This code is provided as is, without any guarantee whatsoever.

  (ns gabrielash.libmisc.printf
    (:require
     [clojure.string :as str ]
     [gabrielash.defnw.core :refer :all]
     [gabrielash.misc.shorts :refer :all]
     [gabrielash.libmisc.core :refer :all]
     [gabrielash.libmisc.string :refer [in?]]))



;;;;;;;;;;;;;;;;;;;;;;;;;; 
;;;;;;;;; SPRINTF
;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- ->int [x]
  #_(println "->int : type of " x " is " (type x))
  (cond (instance? clojure.lang.BigInt x) (biginteger x)
        :else x))

(defn- ->dbl [x]
  (cond (instance? clojure.lang.BigInt x) (double x)
        (instance? clojure.lang.Ratio x) (double x)
        (int? x) (double x)
        :else x))

(defn- ->str [x]
  (cond (nil? x) "nil"
        :else x))



;;;;;;;;;;;;;;;;;;;;

(defnw- cast-argv
  "casts a vector of format arguments to the correct type according to the format string"
  [fmt-str cast-dispatch argv]

  {:let [conversions
         (->> fmt-str
              (re-seq #"%[^a-zA-Z%]*([a-zA-Z%])")
              (map (comp str/lower-case last))
              (remove (*> in? "n%")))]

   :cond [(!= (count conversions) (count argv))
          (throw (ex-info
                  (str "format string <-> arguments mismatch: expected "
                       (count conversions) " args but received " (count argv))
                  {:argv argv :conversions conversions}))]}

  (let [cast-fn (fn [type arg]
                  ((get cast-dispatch type identity) arg))]

    (into []
          (map cast-fn conversions argv))))


(defn sprintf
  "like format, but casts arguments to a type that matches
   the format conversion: 
    * both floating point and integer conversions handle BigIntegers.
    * floating point conversions handle ratios.
    * %s handles nil correctly."
  [fmt-string & argv]
  (let [tag->cast-fn {"d" ->int
                      "x" ->int
                      "o" ->int
                      "f" ->dbl
                      "g" ->dbl
                      "e" ->dbl
                      "a" ->dbl
                      "s" ->str}]
    (apply format
           fmt-string
           (cast-argv fmt-string
                   tag->cast-fn
                   argv))))

(defn printfn
  "like printf, but casts arguments to a type that matches
   the format conversion and adds newline"
  [fmt & argv]
  (println
   (apply sprintf
          fmt
          argv)))

