(ns jackdaw.streams.beta.processors
  (:import
   (org.apache.avro.util Utf8)
   (org.apache.kafka.streams Topology)
   (org.apache.kafka.streams.state Stores)
   (org.apache.kafka.streams.processor Processor ProcessorContext ProcessorSupplier To)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;       BETA API subject to breaking changes!               ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- create-kv-store
  "Return instance of `RocksDBKeyValueStoreSupplier`"
  [store-name key-serde value-serde]
  (-> (Stores/persistentKeyValueStore store-name)
      (Stores/keyValueStoreBuilder key-serde value-serde)))

(defn add-source
  ([builder topic]
   (add-source builder topic (str (:topic-name topic) "-source")))
  ([builder topic source-node-name]
   (.addSource builder
               source-node-name
               (.deserializer (:key-serde topic))
               (.deserializer (:value-serde topic))
               (into-array [(:topic-name topic)]))))

(defn add-sink
  ;; Will use the default Kafka Streams partitioner:
  ;; org.apache.kafka.clients.producer.internals.DefaultPartitioner
  ;; org.apache.kafka.common.utils.Utils.toPositive(Utils.murmur2(keyBytes)) % numPartitions
  ([builder topic parent-node-names]
   (add-sink builder topic (str (:topic-name topic) "-sink") parent-node-names))
  ([builder topic sink-node-name parent-node-names]
   (.addSink builder
             sink-node-name
             (:topic-name topic)
             (.serializer (:key-serde topic))
             (.serializer (:value-serde topic))
             (into-array String parent-node-names))))

(defn add-kv-state-store [builder store]
  (.addStateStore builder
                  (create-kv-store (:store-name store)
                                (:key-serde store)
                                (:value-serde store))
                  ;; Stores are linked when the processor which uses it is added
                  (into-array String [])))

(defn- global-state-store-update-processor [store-fn]
  (let [context (atom nil)]
    (reify Processor
      (close [_])
      (init [_ processor-context]
        (reset! context processor-context))
      (process [_ k message]
        (let [store (store-fn @context)]
          (.put store k message))))))

(defn add-global-state-store [builder store-name source-topic]
  (let [source-node-name (str (name store-name) "-source")
        state-store-name (name store-name)
        processor-node-name (str (name store-name) "-processor")]
    (.addGlobalStore builder
                     ;; persistence for global stores can't have logging (backing topics)
                     ;; enabled, becasue they are backed by their source topic
                     (.withLoggingDisabled
                      (create-kv-store state-store-name
                                    (:key-serde source-topic)
                                    (:value-serde source-topic)))
                     source-node-name
                     (.deserializer (:key-serde source-topic))
                     (.deserializer (:value-serde source-topic))
                     (:topic-name source-topic)
                     processor-node-name
                     (reify ProcessorSupplier
                       (get [_]
                         (global-state-store-update-processor
                          (fn [context]
                            (.getStateStore context state-store-name))))))))

(defn- get-stores [^ProcessorContext ctx state-store-names]
  (reduce (fn [store-map store-name]
            (assoc store-map
                   (keyword store-name)
                   (.getStateStore ctx (name store-name))))
          {} state-store-names))

;; static app configs
(defn- ctx->app-config [^ProcessorContext ctx]
  {:application-id (.applicationId ctx)
   :state-dir (.getCanonicalPath (.stateDir ctx))
   :app-config (into {} (.appConfigs ctx))})

;; dynamic app configs
(defn- ctx->map [^ProcessorContext ctx
                static-ctx
                state-store-names]
  {:ctx-ref ctx
   :processing-ts (System/currentTimeMillis)
   :config (assoc static-ctx
                  :task-id (str (.taskId ctx)))
   :source
   {:topic (.topic ctx)
    :timestamp (.timestamp ctx)
    :partition (.partition ctx)
    :offset (.offset ctx)}
   :stores (get-stores ctx state-store-names)})

(defn- try-to-forward [^ProcessorContext ctx to-node msg-key message]
  (let [topic-name (.topic ctx)]
    (try
     (if (= :all to-node)
       (.forward ctx msg-key message)
       (.forward ctx msg-key message (To/child (name to-node))))
     (catch Throwable e
       (throw (ex-info "Failed to forward message"
                       {:topic to-node
                        :source topic-name
                        :message message}
                       e))))))

(defn- processor-supplier [state-store-names
                          processor-name
                          processor-fn]
  (reify ProcessorSupplier
    (get [_]
      (let [context-ref (atom nil)
            static-ctx (atom {})]
        (reify Processor
          (close [_])
          (init [_ processor-context]
            ;; app config is static, so only bother reading it once
            (reset! static-ctx (ctx->app-config processor-context))
            (reset! context-ref processor-context))
          (process [_ k message]
            (let [processing-result (-> (ctx->map @context-ref
                                                  @static-ctx
                                                  state-store-names)
                                        (processor-fn k message))]
              ;; NOTE: it is important to realise the result fully BEFORE trying to do anything
              ;; with its contents. Otherwise, where processing-result is lazy, errors in the
              ;; processing will not materialise until the sequence is read, which can casue
              ;; partial writes. We want to surface any processing errors first, before forwarding
              ;; anything downstream.
              (doseq [r (doall processing-result)]
                (try-to-forward
                  @context-ref
                  (:forward-to r)
                  (:key r)
                  (:value r)))
              (.commit @context-ref))))))))

(defn add-processor [builder
                     {:keys [processor-name state-store-names parent-node-names] :as processor-config}
                     processor-fn]
  (.addProcessor builder
                 (name processor-name)
                 (processor-supplier state-store-names
                                     processor-name
                                     processor-fn)
                 (into-array String (map name parent-node-names)))
  (.connectProcessorAndStateStores builder
                                   (name processor-name)
                                   (->> state-store-names
                                        (map name)
                                        (into-array String))))
