(ns jackdaw.streams.extras
  "FIXME"
  {:license "BSD 3-Clause License <https://github.com/FundingCircle/jackdaw/blob/master/LICENSE>"}
  (:require [clojure.tools.logging :as log]
            [jackdaw.streams :as js]
            [jackdaw.streams.interop :as jsi]
            [jackdaw.streams.protocols :as jsp]
            [clojure.spec.alpha :as s])
  (:import (jackdaw.streams.interop CljKStream CljKTable CljGlobalKTable CljKGroupedTable
                                    CljKGroupedStream CljTimeWindowedKStream CljSessionWindowedKStream)
           org.apache.kafka.common.TopicPartition
           (org.apache.kafka.streams StreamsBuilder)
           (org.apache.kafka.streams.kstream KStream KTable KGroupedStream KGroupedTable
                                             GlobalKTable TimeWindowedKStream
                                             SessionWindowedKStream Transformer ValueTransformer)
           (org.apache.kafka.streams.processor ProcessorContext StateRestoreListener)
           (org.apache.kafka.streams.state Stores)
           [java.time Instant Duration]))

(set! *warn-on-reflection* true)

(defn transformer-with-ctx [xfm]
   (fn []
     (let [ctx (atom nil)]
       (reify Transformer
         (init [_ processor-context]
           (reset! ctx processor-context))
         (close [_])
         (transform [_ k v]
           (xfm @ctx k v))))))

(defn value-transformer-with-ctx [xfm]
  (fn []
    (let [ctx (atom nil)]
      (reify ValueTransformer
        (init [_ processor-context]
          (reset! ctx processor-context))
        (close [_])
        (transform [_ v]
          (xfm @ctx v))))))

(defn with-stores [store-names xfm]
  (fn [^ProcessorContext ctx & args]
    (let [state-stores (into {} (map (fn [n]
                                       [n (.getStateStore ctx (name n))])) store-names)]
      (apply xfm ctx state-stores args))))

(defn with-state-store
  [builder {:keys [store-name key-serde value-serde] :as store-config}]
  (.addStateStore ^StreamsBuilder (js/streams-builder* builder)
                  (Stores/keyValueStoreBuilder
                    (Stores/persistentKeyValueStore store-name)
                    key-serde
                    value-serde))
  builder)

;; Unwraps the jackdaw class for kafka streams to its Java impl
(defmulti unwrap-kstreams class)
(defmethod unwrap-kstreams CljKStream [ks] (js/kstream* ks))
(defmethod unwrap-kstreams CljKTable [ks] (js/ktable* ks))
(defmethod unwrap-kstreams CljGlobalKTable [ks] (js/global-ktable* ks))
(defmethod unwrap-kstreams CljKGroupedTable [ks] (js/kgroupedtable* ks))
(defmethod unwrap-kstreams CljKGroupedStream [ks] (js/kgroupedstream* ks))
(defmethod unwrap-kstreams CljTimeWindowedKStream [ks] (jsp/time-windowed-kstream* ks))
(defmethod unwrap-kstreams CljSessionWindowedKStream [ks] (jsp/session-windowed-kstream* ks))

;; Wraps the java impl for kafka streams in the jackdaw class
(defmulti wrap-kstreams class)
(defmethod wrap-kstreams KStream [ks] (jsi/clj-kstream ks))
(defmethod wrap-kstreams KTable [ks] (jsi/clj-ktable ks))
(defmethod wrap-kstreams KGroupedStream [ks] (jsi/clj-kgroupedstream ks))
(defmethod wrap-kstreams KGroupedTable [ks] (jsi/clj-kgroupedtable ks))
(defmethod wrap-kstreams GlobalKTable [ks] (jsi/clj-global-ktable ks))
(defmethod wrap-kstreams TimeWindowedKStream [ks] (jsi/clj-time-windowed-kstream ks))
(defmethod wrap-kstreams SessionWindowedKStream [ks] (jsi/clj-session-windowed-kstream ks))

(defn with-java-api
  "Given a clojure/jackdaw kafka streams wrapper unwraps it, calls f
  passing the java kafka streams object and then re-wraps the result
  with the clojure/jackdaw wrapper. Allows you to 'drop down' to the
  java API within a Jackdaw kafka streams topology builder."
  [ks f]
  (wrap-kstreams (f (unwrap-kstreams ks))))

(defn map-validating!
  [builder topic topic-spec {:keys [line file]}]
  (js/map-values builder
                 (fn [val]
                   (if-not (s/valid? topic-spec val)
                     (let [msg "Failed to validate outbound record!"
                           data (merge {::topic (:topic.metadata/name topic)
                                        ::line  line
                                        ::file  file}
                                       (s/explain-data topic-spec val))]
                       (log/fatal msg (pr-str data))
                       (throw (ex-info msg data)))
                     val))))

(defn with-file [form-meta]
  (update form-meta :file #(or %1 *file*)))

(defn logging-state-restore-listener
  "Returns a new Kafka `StateRestoreListener` instance which logs when
  batches are restored, and how long it takes to restore all the
  batches for a given partition."
  ^StateRestoreListener []
  (let [restore-tracker (atom {})]
    (reify StateRestoreListener
      (^void onRestoreStart [_
                             ^TopicPartition topicPartition,
                             ^String storeName
                             ^long startingOffset
                             ^long endingOffset]
       (swap! restore-tracker assoc storeName (Instant/now))
       (log/warnf "Restoring state store (%s.%d) over offsets %s...%s"
                  (.topic topicPartition) (.partition topicPartition)
                  startingOffset endingOffset))

      (^void onBatchRestored [_
                              ^TopicPartition topicPartition
                              ^String storeName
                              ^long batchEndOffset
                              ^long numRestored]
       (log/warnf "Restored a batch from (%s.%d)"
                  (.topic topicPartition) (.partition topicPartition)))
      (^void onRestoreEnd [_
                           ^TopicPartition topicPartition
                           ^String storeName
                           ^long totalRestored]
       (let [start-date  (get @restore-tracker storeName)
             elapsed-sec (.getSeconds (Duration/between start-date
                                                        (Instant/now)))]
         (log/warnf "Finished restoring state store (%s.%d) elapsed %s"
                    (.topic topicPartition) (.partition topicPartition)
                    elapsed-sec))))))

(defmacro to!
  "Wraps `#'jackdaw.streams/to!`, providing validation of records
  against the spec of the to! topic."
  ([builder topic topic-spec]
   `(let [t# ~topic]
      (-> ~builder
          (map-validating! t# ~topic-spec ~(with-file (meta &form)))
          (js/to t#))))
  ([builder partition-fn topic topic-spec]
   `(let [t# ~topic]
      (-> ~builder
          (map-validating! t# ~topic-spec ~(with-file (meta &form)))
          (js/to ~partition-fn t#)))))

(defmacro through
  "Wraps `#'jackdaw.streams/through`, providing validation of records
  against the spec of the through topic."
  ([builder topic topic-spec]
   `(let [t# ~topic]
      (-> ~builder
          (map-validating! t# ~topic-spec ~(with-file (meta &form)))
          (js/through t#))))
  ([builder partition-fn topic topic-spec]
   `(let [t# ~topic]
      (-> ~builder
          (map-validating! t# ~topic-spec ~(with-file (meta &form)))
          (js/through ~partition-fn t#)))))
