(ns jackdaw.streams.xform
  "Helper functions for working with transducers."
  (:gen-class)
  (:require [jackdaw.serdes :as js]
            [jackdaw.streams :as j])
  (:import org.apache.kafka.streams.kstream.ValueTransformer
           [org.apache.kafka.streams.state KeyValueStore Stores]
           org.apache.kafka.streams.StreamsBuilder))

(defn fake-kv-store
  "Creates an instance of org.apache.kafka.streams.state.KeyValueStore
  with overrides for get and put."
  [init]
  (let [store (volatile! init)]
    (reify KeyValueStore
      (get [_ k]
        (clojure.core/get @store k))

      (put [_ k v]
        (vswap! store assoc k v)))))

(defn kv-store-swap-fn
  "Takes an instance of KeyValueStore, a function f, and a map m, and
  updates the store in a manner similar to `clojure.core/swap!`."
  [^KeyValueStore store f m]
  (let [prev (reduce (fn [m k]
                       (assoc m k (.get store k)))
                     {}
                     (keys m))
        next (f prev m)]
    (doall (map (fn [[k v]] (.put store k v)) next))
    next))

(defn value-transformer
  "Creates an instance of org.apache.kafka.streams.kstream.ValueTransformer
  with overrides for init, transform, and close."
  [xf]
  (let [ctx (atom nil)]
    (reify
      ValueTransformer
      (init [_ context]
        (reset! ctx context))
      (transform [_ v]
        (let [^KeyValueStore store (.getStateStore @ctx "transducer")]
          (first (into [] (xf store) [v]))))
      (close [_]))))

(defn transduce-kstream
  [kstream xf]
  "Takes kstream and xf and transduces the stream."
  (-> kstream
      (j/transform-values (fn [] (value-transformer xf)) ["transducer"])
      (j/flat-map (fn [[_ v]] v))))

(defn add-state-store!
  [builder]
  "Takes a builder and adds a state store."
  (doto ^StreamsBuilder (j/streams-builder* builder)
    (.addStateStore (Stores/keyValueStoreBuilder
                     (Stores/persistentKeyValueStore "transducer")
                     (js/edn-serde)
                     (js/edn-serde))))
  builder)
