[![Clojars Project](https://img.shields.io/clojars/v/functionalbytes/rmap.svg)](https://clojars.org/functionalbytes/rmap)
[![cljdoc badge](https://cljdoc.org/badge/functionalbytes/rmap)](https://cljdoc.org/d/functionalbytes/rmap/CURRENT)
[![Clojure CI](https://github.com/aroemers/rmap/workflows/Clojure%20CI/badge.svg?branch=master)](https://github.com/aroemers/rmap/actions?query=workflow%3A%22Clojure+CI%22)
[![Clojars Project](https://img.shields.io/clojars/dt/functionalbytes/rmap?color=blue)](https://clojars.org/functionalbytes/rmap)
![Map](https://img.shields.io/badge/map-recursive-brightgreen)

# ➰ rmap

A Clojure library for literal recursive maps.

![Banner](banner.png)

## Usage

### The RVal object

We start with a basic building block: a recursive value in the form of an RVal object.
You can create an RVal using the `rval` macro.
It takes a body of expressions, which will not be evaluated yet.
We call this a recursive value, because the body has access to a `ref` function.
By using this `ref` function, it can fetch other values from the associative datastructure - i.e a map or a vector - it is "evaluated" in.

Let's create a simple map with an RVal and print it:

```clj
(def basic {:foo 1
            :bar (rval (println "Calculating bar...")
                       (inc (ref :foo)))})
;=> #'user/basic

basic
;=> {:foo 1, :bar ??}
```

As you can see, the `:bar` entry is an RVal and uses the `ref` function to fetch the value mapped to `:foo`.
You can also see that the `:bar` entry is not evaluated yet.

### The ref function

To evaluate an RVal, it needs a `ref` function for a particular context.
We can create such a function by passing the map or vector to `->ref`.
The resulting function can fetch values from the given map or vector.
If that value is an RVal, the function will evaluate it _passing itself_.
Recursion! 💥

Let's create a `ref` function for the basic map we created before and use it:

```clj
(def ref (->ref basic))
;=> #'user/ref

(ref :b)
Calculating bar...
;=> 2

(ref :b)
;=> 2
```

You can see that the `:bar` entry is evaluated now, yielding the expected result.
You can also see that the result is cached, as the "calculating" message is only printed once.
This caching happens on the level of the created `ref` function.
The original map itself has not changed:

```clj
basic
;=> {:foo 1, :bar ??}
```

### Combining the building blocks

To make it easer to create recursive maps, another macro is provided, called `rmap`.
This takes a literal map or vector and wraps each of the value expressions as RVal objects.
For example:

```clj
(def root
  (rmap {:a 1
         :b (inc (ref :a))
         :c (+ (ref :b) (ref :d 40)}))
;=> #'user/root

root
;=> {:a ?? :b ?? :c ??}
```

You can see that all entries are RVals.
Also note how the `:c` entry passes a "not-found" parameter to the `ref` function in the value expressions.

Instead of creating a `ref` function and fetching values with it, there is another way of getting the evaluated values.
By calling `finalize` on a map or vector it is "updated" with all entries evaluated.
For example:

```clj
(finalize root)
;=> {:a 1 :b 2 :c 42}

(finalize (assoc root :a 1001))
;=> {:a 1001 :b 1002 :c 1042}
```

Now you can work on the map like you're used to.
Because it is common to work on a finalized recursive map, you can use `rmap!` to combine `rmap` and `finalize`.
For example:

```clj
(rmap! {:a 1
        :b (inc (ref :a))
        :c (inc (ref :c))})
;=> {:a 1 :b 2 :c 3}
```

_That's it. Enjoy!_ 🚀

## License

Copyright © 2014-2020 Functional Bytes

Distributed under the Eclipse Public License either version 1.0 or (at
your option) any later version.
