package funcatron.intf;

import java.util.Map;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Implement this Interface for a Funcatron Endpoint.
 */
public interface Func<Request> {
    /**
     * For each incoming request, a new `Func` instance is created and the `apply`
     * method is invoked.
     *
     * If the `Request` type is `Object` or `Map`, then the JSON body
     * of the incoming request is passed directly to the method.
     *
     * If the `Request` type is `String` or `byte[]`, then the raw request is passed in.
     *
     * If the `Request` type is `Node` or `Document`, then the request body is parsed using a DOM XML parser
     *
     * If the request method (found in the Context) is GET or DELETE, the `req` is null
     *
     * Regardless of the declared return type, here's the translation:
     * byte[] or String -- passed back
     * Node or Document -- considered XML, serialized to bytes accordingly
     * Map or anything that's not listed -- JSON serialize
     * MetaResponse -- the contents of this object will determine the response
     *
     * @param req the incoming request. `null` if the request is a GET or DELETE
     *
     * @param context
     * @return
     */
    Object apply(Request req, Context context);

    /**
     * A function that will take the JSON data in a Map and convert it
     * into a Request object. This is a good place to set up a Jackson
     * parser with custom Jackson stuff rather than relying on the bare
     * bones Jackson parser in the Runner.
     *
     * @return a customer JSON to Object parser or null if Funcatron should use the default Jackson implementation
     */
    default Function<Map<String, Object>, Request> jsonDecoder() {
        return null;
    }


    /**
     * A customer object serializer that will take a return value and JSON serialize it as a byte array. This is
     * a good place use a custom Jackson serializer. If null is returned, then use the default Funcatron Jackson
     * implementation
     *
     * @return The object to JSON serializer or null if there's no customer serializer
     */
    default Function<Object, byte[]> jsonEncoder() {
        return null;
    }
}
