(ns fulcro-spec.signature
  "On-demand signature computation for staleness detection.

   Signatures are short hashes of normalized function source code.
   When a function's implementation changes, its signature changes,
   allowing detection of stale test coverage.

   The signature is computed by:
   1. Extracting full source using clojure.repl/source-fn
   2. Removing docstrings (so doc changes don't invalidate tests)
   3. Normalizing whitespace (so formatting changes don't invalidate tests)
   4. Hashing with SHA-256 and taking first 6 characters

   Two-field transitive signatures:
   The `transitive-signature` function returns a signature in the format
   'self,callees' where:
   - self: 6-char hash of the function's own source
   - callees: 6-char hash of all transitive callees' signatures (sorted)

   This enables detecting changes to both the function itself AND any
   functions it calls transitively.

   Performance options (Java system properties):
   - fulcro.spec.skip-checked: Enable auto-skipping of already-checked tests
   - fulcro.spec.cache-signatures: Cache signatures for duration of JVM"
  (:require
    [clojure.repl]
    [clojure.string :as str]
    [com.fulcrologic.guardrails.impl.externs :as gr.externs])
  (:import
    (java.nio.charset StandardCharsets)
    (java.security MessageDigest)))

;; =============================================================================
;; Content Normalization
;; =============================================================================

(defn- find-string-end
  "Finds the end position of a string starting at idx (after opening quote).
   Returns the index after the closing quote, or nil if not found.
   Handles escape sequences correctly."
  [s idx]
  (loop [i idx]
    (when (< i (count s))
      (let [ch (get s i)]
        (cond
          (= ch \\) (recur (+ i 2))  ; Skip escaped char
          (= ch \") (inc i)          ; Found closing quote
          :else (recur (inc i)))))))

(defn- skip-whitespace
  "Returns index of first non-whitespace character starting from idx."
  [s idx]
  (loop [i idx]
    (if (and (< i (count s))
          (Character/isWhitespace (char (get s i))))
      (recur (inc i))
      i)))

(defn- find-matching-bracket
  "Finds the closing bracket matching the opening bracket at idx.
   Returns index after the closing bracket, or nil if not found.
   Handles nested brackets and strings correctly."
  [s idx]
  (let [open-ch  (get s idx)
        close-ch (case open-ch
                   \[ \]
                   \( \)
                   \{ \}
                   nil)]
    (when close-ch
      (loop [i     (inc idx)
             depth 1]
        (when (< i (count s))
          (let [ch (get s i)]
            (cond
              (= ch \")
              (if-let [end (find-string-end s (inc i))]
                (recur end depth)
                nil)

              (= ch open-ch)
              (recur (inc i) (inc depth))

              (= ch close-ch)
              (if (= depth 1)
                (inc i)
                (recur (inc i) (dec depth)))

              :else
              (recur (inc i) depth))))))))

(defn- remove-docstring-from-def
  "Removes docstring from a def* form (defn, defn-, >defn, etc.) in source text.

   Handles both forms:
   - (defn name \"docstring\" [args] body)
   - (defn name [args] \"docstring\" body)

   Returns source with docstrings removed."
  [s]
  (let [len (count s)]
    (loop [i      0
           result (StringBuilder.)]
      (if (>= i len)
        (str result)
        (let [ch (get s i)]
          (cond
            ;; Handle strings not in def position - preserve them
            (= ch \")
            (if-let [end (find-string-end s (inc i))]
              (do
                (.append result (subs s i end))
                (recur end result))
              (recur (inc i) result))

            ;; Look for def forms: (def..., (>def...
            (and (= ch \()
              (< (inc i) len)
              (let [next-ch (get s (inc i))]
                (or (= next-ch \d)           ; (def...
                    (= next-ch \>))))        ; (>def...
            (let [;; Skip the > if present
                  def-start (if (= \> (get s (inc i))) (+ i 2) (inc i))
                  def-end (loop [j def-start]
                            (if (and (< j len)
                                  (let [c (get s j)]
                                    (or (Character/isLetterOrDigit (char c))
                                      (= c \-)
                                      (= c \>))))  ; for >defn
                              (recur (inc j))
                              j))]
              (if (and (>= (- def-end def-start) 3)  ; At least "def"
                    (str/starts-with? (subs s def-start (min (+ def-start 3) len)) "def")
                    (< def-end len)
                    (Character/isWhitespace (get s def-end)))
                (let [after-def  (skip-whitespace s def-end)
                      ;; Skip the function name
                      name-end   (loop [j after-def]
                                   (if (and (< j len)
                                         (let [c (get s j)]
                                           (not (Character/isWhitespace (char c)))))
                                     (recur (inc j))
                                     j))
                      after-name (skip-whitespace s name-end)]
                  ;; Check for docstring right after name: (defn name "doc" ...)
                  (if (and (< after-name len)
                        (= \" (get s after-name)))
                    (if-let [doc-end (find-string-end s (inc after-name))]
                      (do
                        (.append result (subs s i after-name))
                        (recur doc-end result))
                      (do
                        (.append result ch)
                        (recur (inc i) result)))
                    ;; Check for docstring after args: (defn name [args] "doc" ...)
                    (if (and (< after-name len)
                          (= \[ (get s after-name)))
                      (if-let [args-end (find-matching-bracket s after-name)]
                        (let [after-args (skip-whitespace s args-end)]
                          (if (and (< after-args len)
                                (= \" (get s after-args)))
                            (if-let [doc-end (find-string-end s (inc after-args))]
                              (do
                                (.append result (subs s i after-args))
                                (recur doc-end result))
                              (do
                                (.append result ch)
                                (recur (inc i) result)))
                            (do
                              (.append result ch)
                              (recur (inc i) result))))
                        (do
                          (.append result ch)
                          (recur (inc i) result)))
                      (do
                        (.append result ch)
                        (recur (inc i) result)))))
                (do
                  (.append result ch)
                  (recur (inc i) result))))

            :else
            (do
              (.append result ch)
              (recur (inc i) result))))))))

(defn- normalize-content
  "Normalizes source code content for semantic comparison.

   Removes docstrings and normalizes whitespace so that:
   - Implementation changes ARE detected
   - Docstring changes are NOT detected
   - Whitespace/formatting changes are NOT detected"
  [source-text]
  (when source-text
    (try
      (let [without-docs (remove-docstring-from-def source-text)
            normalized   (-> without-docs
                           (str/replace #"\s+" " ")
                           str/trim)]
        normalized)
      (catch Exception _e
        source-text))))

;; =============================================================================
;; Hashing
;; =============================================================================

(defn- sha256
  "Generates a SHA256 hash of the input string."
  [^String s]
  (when s
    (let [digest     (MessageDigest/getInstance "SHA-256")
          hash-bytes (.digest digest (.getBytes s StandardCharsets/UTF_8))]
      (apply str (map #(format "%02x" %) hash-bytes)))))

(defn- hash-content
  "Generates a content hash for normalized source text.
   Returns full SHA256 hex string."
  [source-text]
  (some-> source-text
    normalize-content
    sha256))

;; =============================================================================
;; Public API
;; =============================================================================

(defn signature
  "Computes a short signature (first 6 chars of SHA256) for a function.

   Args:
     fn-sym - Fully qualified symbol of the function

   Returns:
     6-character signature string, or nil if:
     - Function cannot be resolved
     - Source is not available (e.g., compiled without source)
     - Function was defined at REPL without file

   The signature is based on the normalized source code of the function,
   with docstrings removed and whitespace normalized. This means:
   - Changing implementation DOES change the signature
   - Changing docstrings does NOT change the signature
   - Reformatting whitespace does NOT change the signature

   Example:
     (signature 'myapp.orders/create-order)
     ;; => \"a1b2c3\""
  [fn-sym]
  (when-let [source (clojure.repl/source-fn fn-sym)]
    (when-let [hash (hash-content source)]
      (subs hash 0 (min 6 (count hash))))))

;; =============================================================================
;; Property Checks (cached at first access for zero overhead)
;; =============================================================================

(def ^:private skip-checked-enabled?*
  "Cached result of skip-checked property check."
  (delay (= "true" (System/getProperty "fulcro.spec.skip-checked"))))

(def ^:private cache-signatures-enabled?*
  "Cached result of cache-signatures property check."
  (delay (= "true" (System/getProperty "fulcro.spec.cache-signatures"))))

(defn skip-checked-enabled?
  "Returns true if the fulcro.spec.skip-checked system property is 'true'.
   Result is cached at first call for zero overhead on subsequent checks."
  []
  @skip-checked-enabled?*)

(defn cache-signatures-enabled?
  "Returns true if the fulcro.spec.cache-signatures system property is 'true'.
   Result is cached at first call for zero overhead on subsequent checks."
  []
  @cache-signatures-enabled?*)

;; =============================================================================
;; Signature Caching (only used when cache-signatures enabled)
;; =============================================================================

;; Cache for computed signatures. Only populated when cache-signatures is enabled.
;; Valid for duration of JVM process - intended for full test suite runs,
;; not interactive development where code changes between test runs.
(defonce ^:private signature-cache (atom {}))

(defn- get-signature
  "Gets signature for fn-sym, using cache if -Dfulcro.spec.cache-signatures=true."
  [fn-sym]
  (if (cache-signatures-enabled?)
    (or (get @signature-cache fn-sym)
        (let [sig (signature fn-sym)]
          (swap! signature-cache assoc fn-sym sig)
          sig))
    (signature fn-sym)))

;; =============================================================================
;; Transitive Signatures
;; =============================================================================

(defn transitive-signature
  "Returns a two-field signature 'self,callees' for a function.

   Format: \"xxxxxx,yyyyyy\"
   - xxxxxx: 6-char hash of the function's own source
   - yyyyyy: 6-char hash of all transitive callees' signatures (sorted)
             or \"000000\" for leaf functions with no callees

   This enables detecting changes to both the function itself AND any
   functions it calls transitively within the given scope.

   Args:
     fn-sym - Fully qualified symbol of the function
     scope-ns-prefixes - Set of namespace prefix strings to include

   Uses signature caching if -Dfulcro.spec.cache-signatures=true

   Example:
     (transitive-signature 'myapp.orders/process #{\"myapp\"})
     ;; => \"a1b2c3,d4e5f6\""
  [fn-sym scope-ns-prefixes]
  (when-let [self-sig (get-signature fn-sym)]
    (let [all-callees (gr.externs/transitive-calls fn-sym scope-ns-prefixes)
          callees     (disj all-callees fn-sym)
          sorted      (sort-by str callees)
          callee-sigs (keep get-signature sorted)
          combined    (str/join "," callee-sigs)
          callees-sig (if (seq callee-sigs)
                        (subs (sha256 combined) 0 6)
                        "000000")]
      (str self-sig "," callees-sig))))

;; =============================================================================
;; Test Skipping Predicate
;; =============================================================================

(defn already-checked?
  "Returns true if this test's coverage declarations indicate it has already
   been verified and nothing has changed since.

   IMPORTANT: Zero overhead when disabled - property check short-circuits first.
   No expensive signature computation occurs unless skip-checked is enabled.

   A test is considered 'already checked' when:
   1. -Dfulcro.spec.skip-checked=true
   2. scope-ns-prefixes configured (passed as argument)
   3. Every covered function's sealed signature matches current transitive signature

   This means: the test was previously run, passed, and the developer sealed
   it by recording the signatures in the :covers metadata. Since then, neither
   the covered functions nor any functions they call have changed.

   For fast full-suite runs, also enable -Dfulcro.spec.cache-signatures=true

   Args:
     covers-map - Map of {fn-symbol sealed-signature} from test's :covers metadata
     scope-ns-prefixes - Set of namespace prefix strings for transitive analysis

   Returns:
     true if test can be skipped, false otherwise"
  [covers-map scope-ns-prefixes]
  ;; Property check FIRST - returns false immediately if disabled (zero overhead)
  (and (skip-checked-enabled?)
       (seq scope-ns-prefixes)
       (every? (fn [[fn-sym sealed-sig]]
                 (= sealed-sig (transitive-signature fn-sym scope-ns-prefixes)))
               covers-map)))
