(ns fulcro-spec.signature
  "On-demand signature computation for staleness detection.

   Signatures are short hashes of normalized function source code.
   When a function's implementation changes, its signature changes,
   allowing detection of stale test coverage.

   Based on techniques from com.fulcrologic.test-filter.content."
  (:require
    [clojure.string :as str])
  (:import
    (java.nio.charset StandardCharsets)
    (java.security MessageDigest)))

;; =============================================================================
;; Source Text Extraction
;; =============================================================================

(defn- read-file-lines
  "Reads a file and returns a vector of lines (1-indexed for convenience)."
  [file-path]
  (vec (cons nil (str/split-lines (slurp file-path)))))

(defn- extract-source-text
  "Extracts source text from line range."
  [file-path start-line end-line]
  (try
    (let [lines (read-file-lines file-path)]
      (when (and start-line end-line
              (<= start-line (dec (count lines)))
              (<= end-line (dec (count lines))))
        (str/join "\n"
          (subvec lines start-line (inc end-line)))))
    (catch Exception _e
      nil)))

;; =============================================================================
;; Content Normalization
;; =============================================================================

(defn- find-string-end
  "Finds the end position of a string starting at idx (after opening quote).
  Returns the index after the closing quote, or nil if not found."
  [s idx]
  (loop [i idx]
    (when (< i (count s))
      (let [ch (get s i)]
        (cond
          (= ch \\) (recur (+ i 2))
          (= ch \") (inc i)
          :else (recur (inc i)))))))

(defn- skip-whitespace
  "Returns index of first non-whitespace character starting from idx."
  [s idx]
  (loop [i idx]
    (if (and (< i (count s))
          (Character/isWhitespace (char (get s i))))
      (recur (inc i))
      i)))

(defn- find-matching-bracket
  "Finds the closing bracket matching the opening bracket at idx."
  [s idx]
  (let [open-ch  (get s idx)
        close-ch (case open-ch
                   \[ \]
                   \( \)
                   \{ \}
                   nil)]
    (when close-ch
      (loop [i     (inc idx)
             depth 1]
        (when (< i (count s))
          (let [ch (get s i)]
            (cond
              (= ch \")
              (if-let [end (find-string-end s (inc i))]
                (recur end depth)
                nil)

              (= ch open-ch)
              (recur (inc i) (inc depth))

              (= ch close-ch)
              (if (= depth 1)
                (inc i)
                (recur (inc i) (dec depth)))

              :else
              (recur (inc i) depth))))))))

(defn- remove-docstring-from-def
  "Removes docstring from a def* form in source text."
  [s]
  (let [len (count s)]
    (loop [i      0
           result (StringBuilder.)]
      (if (>= i len)
        (str result)
        (let [ch (get s i)]
          (cond
            (= ch \")
            (if-let [end (find-string-end s (inc i))]
              (do
                (.append result (subs s i end))
                (recur end result))
              (recur (inc i) result))

            (and (= ch \()
              (< (inc i) len)
              (= \d (get s (inc i))))
            (let [def-end (loop [j (inc i)]
                            (if (and (< j len)
                                  (let [c (get s j)]
                                    (or (Character/isLetterOrDigit (char c))
                                      (= c \-))))
                              (recur (inc j))
                              j))]
              (if (and (str/starts-with? (subs s (inc i) def-end) "def")
                    (< def-end len)
                    (Character/isWhitespace (get s def-end)))
                (let [after-def  (skip-whitespace s def-end)
                      name-end   (loop [j after-def]
                                   (if (and (< j len)
                                         (let [c (get s j)]
                                           (not (Character/isWhitespace (char c)))))
                                     (recur (inc j))
                                     j))
                      after-name (skip-whitespace s name-end)]
                  (if (and (< after-name len)
                        (= \" (get s after-name)))
                    (if-let [doc-end (find-string-end s (inc after-name))]
                      (do
                        (.append result (subs s i after-name))
                        (recur doc-end result))
                      (do
                        (.append result ch)
                        (recur (inc i) result)))
                    (if (and (< after-name len)
                          (= \[ (get s after-name)))
                      (if-let [args-end (find-matching-bracket s after-name)]
                        (let [after-args (skip-whitespace s args-end)]
                          (if (and (< after-args len)
                                (= \" (get s after-args)))
                            (if-let [doc-end (find-string-end s (inc after-args))]
                              (do
                                (.append result (subs s i after-args))
                                (recur doc-end result))
                              (do
                                (.append result ch)
                                (recur (inc i) result)))
                            (do
                              (.append result ch)
                              (recur (inc i) result))))
                        (do
                          (.append result ch)
                          (recur (inc i) result)))
                      (do
                        (.append result ch)
                        (recur (inc i) result)))))
                (do
                  (.append result ch)
                  (recur (inc i) result))))

            :else
            (do
              (.append result ch)
              (recur (inc i) result))))))))

(defn- normalize-content
  "Normalizes source code content for semantic comparison.
  Removes docstrings and normalizes whitespace."
  [source-text]
  (when source-text
    (try
      (let [without-docs (remove-docstring-from-def source-text)
            normalized   (-> without-docs
                           (str/replace #"\s+" " ")
                           str/trim)]
        normalized)
      (catch Exception _e
        source-text))))

;; =============================================================================
;; Hashing
;; =============================================================================

(defn- sha256
  "Generates a SHA256 hash of the input string."
  [^String s]
  (when s
    (let [digest     (MessageDigest/getInstance "SHA-256")
          hash-bytes (.digest digest (.getBytes s StandardCharsets/UTF_8))]
      (apply str (map #(format "%02x" %) hash-bytes)))))

(defn- hash-content
  "Generates a content hash for normalized source text."
  [source-text]
  (some-> source-text
    normalize-content
    sha256))

;; =============================================================================
;; Public API
;; =============================================================================

(defn compute-signature
  "Computes a short signature (first 6 chars of SHA256) for a function.

   Args:
     fn-sym - Fully qualified symbol of the function

   Returns:
     6-character signature string, or nil if computation fails.

   The signature is based on the normalized source code of the function,
   with docstrings removed and whitespace normalized. This means:
   - Changing implementation changes the signature
   - Changing docstrings does NOT change the signature
   - Reformatting whitespace does NOT change the signature"
  [fn-sym]
  (when-let [v (resolve fn-sym)]
    (let [m (meta v)
          file (:file m)
          line (:line m)
          end-line (or (:end-line m) line)]
      (when (and file line)
        ;; Need to find the actual file path
        (let [file-path (if (.startsWith ^String file "/")
                          file
                          (when-let [resource (clojure.java.io/resource file)]
                            (.getPath resource)))]
          (when file-path
            (when-let [source (extract-source-text file-path line end-line)]
              (when-let [hash (hash-content source)]
                (subs hash 0 6)))))))))

(defn signature
  "Returns the current signature for a function symbol.
   Convenience wrapper around compute-signature.

   Usage:
     (signature 'myapp.orders/create-order)
     ;; => \"a1b2c3\""
  [fn-sym]
  (compute-signature fn-sym))
