(ns fulcro-spec.proof
  "Tools for verifying transitive test coverage proofs.

   This namespace provides functions to verify that a function and all its
   transitive dependencies have declared test coverage. This enables building
   a provable chain of tests from low-level functions up to application logic.

   Additionally, it supports staleness detection: when a function's source code
   changes after a test was sealed, the coverage is marked as stale until the
   developer reviews and re-seals the test.

   Configuration:
   Use `configure!` to set global options once for your project:
   - :scope-ns-prefixes - set of namespace prefix strings to include (e.g. #{\"myapp\"})
   - :enforce? - when true, when-mocking!! and provided!! will throw on missing coverage

   Example:
     (proof/configure! {:scope-ns-prefixes #{\"myapp\" \"myapp.lib\"}
                        :enforce? true})"
  (:require
    [fulcro-spec.coverage :as cov]
    #?(:clj [fulcro-spec.signature :as sig])
    #?(:clj [com.fulcrologic.guardrails.impl.externs :as gr.externs])
    #?(:clj [clojure.set :as set])))

;; =============================================================================
;; Global Configuration
;; =============================================================================

(defonce ^:dynamic *config*
  (atom {:scope-ns-prefixes #{}
         :enforce? false}))

(defn configure!
  "Configure global proof settings. Options:
   - :scope-ns-prefixes - Set of namespace prefix strings that define which
                          namespaces are 'in scope' for coverage checking.
                          E.g., #{\"myapp\"} includes myapp.core, myapp.db, etc.
   - :enforce? - When true, when-mocking!! and provided!! will throw exceptions
                 if mocked functions lack transitive coverage. When false, they
                 behave like when-mocking!/provided!. Default: false.

   Call this once at the start of your test suite or in a test fixture."
  [opts]
  (swap! *config* merge opts))

(defn get-config
  "Returns the current proof configuration."
  []
  @*config*)

(defn- effective-scope
  "Returns the scope-ns-prefixes to use, preferring explicit opts over global config."
  [opts]
  (or (:scope-ns-prefixes opts)
      (:scope-ns-prefixes @*config*)))

(defn- enforcement-enabled?
  "Returns whether proof enforcement is enabled."
  []
  (:enforce? @*config*))

;; =============================================================================
;; Signature and Freshness
;; =============================================================================

#?(:clj
   (defn signature
     "Returns the current 6-character signature for a function.

      The signature is a hash of the function's normalized source code.
      Use this to get the signature when sealing a test.

      Example:
        (signature 'myapp.orders/create-order)
        ;; => \"a1b2c3\""
     [fn-sym]
     (sig/signature fn-sym)))

#?(:clj
   (defn fresh?
     "Returns true if the function's current signature matches its sealed signature.

      A function is fresh if:
      - It has no sealed signature (not yet sealed), OR
      - Its current signature matches the sealed signature

      A function is stale if it has a sealed signature that differs from current."
     [fn-sym]
     (let [sealed (cov/sealed-signature fn-sym)]
       (or (nil? sealed)
           (= sealed (sig/signature fn-sym))))))

#?(:clj
   (defn stale?
     "Returns true if the function has a sealed signature that differs from current.

      A stale function means its implementation has changed since the covering
      test was verified. The test needs review and re-sealing."
     [fn-sym]
     (let [sealed (cov/sealed-signature fn-sym)]
       (and (some? sealed)
            (not= sealed (sig/signature fn-sym))))))

;; =============================================================================
;; Core Verification Functions
;; =============================================================================

(defn verify-transitive-coverage
  "Verify that fn-sym and all its transitive dependencies have test coverage.

   Returns a report map:
   - :function - the function checked
   - :scope-ns-prefixes - the namespace scope used
   - :transitive-deps - all guardrailed functions in the call graph
   - :covered - set of functions with declared coverage
   - :uncovered - set of functions without declared coverage
   - :stale - set of functions with stale coverage (signature mismatch)
   - :proof-complete? - true if all deps are covered AND none are stale

   Options (optional if global config is set):
   - :scope-ns-prefixes - set of namespace prefix strings to include"
  ([fn-sym] (verify-transitive-coverage fn-sym {}))
  ([fn-sym opts]
   #?(:clj
      (let [scope (effective-scope opts)]
        (if (empty? scope)
          {:function fn-sym
           :scope-ns-prefixes scope
           :transitive-deps #{}
           :covered #{}
           :uncovered #{}
           :stale #{}
           :proof-complete? true
           :note "No scope configured - use configure! or pass :scope-ns-prefixes"}
          (let [all-deps (gr.externs/transitive-calls fn-sym scope)
                covered (into #{} (filter cov/covered? all-deps))
                uncovered (set/difference all-deps covered)
                stale-fns (into #{} (filter stale? covered))]
            {:function fn-sym
             :scope-ns-prefixes scope
             :transitive-deps all-deps
             :covered covered
             :uncovered uncovered
             :stale stale-fns
             :proof-complete? (and (empty? uncovered) (empty? stale-fns))})))
      :cljs
      {:function fn-sym
       :scope-ns-prefixes (effective-scope opts)
       :transitive-deps #{}
       :covered #{}
       :uncovered #{}
       :stale #{}
       :proof-complete? true
       :note "Call graph analysis not available in ClojureScript"})))

(defn fully-tested?
  "Returns true if the function and all its transitive dependencies have
   declared test coverage within the configured scope AND none are stale.

   This is the simple query function for checking coverage status.
   Configure scope first with `configure!` or pass opts.

   Examples:
     (fully-tested? 'myapp.orders/create-order)
     (fully-tested? 'myapp.orders/create-order {:scope-ns-prefixes #{\"myapp\"}})"
  ([fn-sym] (fully-tested? fn-sym {}))
  ([fn-sym opts]
   (:proof-complete? (verify-transitive-coverage fn-sym opts))))

(defn why-not-tested?
  "Returns a map of issues if fn-sym is not fully tested, or nil if fully tested.

   The returned map contains:
   - :uncovered - set of functions without declared coverage
   - :stale - set of functions with stale coverage (signature mismatch)

   Examples:
     (why-not-tested? 'myapp.orders/create-order)
     ;; => {:uncovered #{myapp.db/save!} :stale #{myapp.validation/check}}"
  ([fn-sym] (why-not-tested? fn-sym {}))
  ([fn-sym opts]
   (let [{:keys [proof-complete? uncovered stale]} (verify-transitive-coverage fn-sym opts)]
     (when-not proof-complete?
       (cond-> {}
         (seq uncovered) (assoc :uncovered uncovered)
         (seq stale) (assoc :stale stale))))))

;; =============================================================================
;; Assertion and Enforcement
;; =============================================================================

(defn assert-transitive-coverage!
  "Assert that fn-sym has complete transitive coverage (no uncovered or stale deps).
   Throws an exception if any in-scope dependency lacks coverage or is stale.
   Used by when-mocking!! at test time.

   When called without opts, uses global config.
   Only throws if :enforce? is true in config (or if opts explicitly passed)."
  ([fn-sym] (assert-transitive-coverage! fn-sym {}))
  ([fn-sym opts]
   #?(:clj
      (let [scope (effective-scope opts)
            ;; If opts were explicitly passed with scope, always enforce
            ;; Otherwise respect the global :enforce? flag
            should-enforce? (or (seq (:scope-ns-prefixes opts))
                               (enforcement-enabled?))]
        (when (and should-enforce? (seq scope))
          (let [{:keys [uncovered stale proof-complete?]} (verify-transitive-coverage fn-sym opts)]
            (when-not proof-complete?
              (throw (ex-info (str "Incomplete transitive coverage for " fn-sym
                                ". Uncovered: " (pr-str uncovered)
                                ". Stale: " (pr-str stale))
                       {:function fn-sym
                        :uncovered uncovered
                        :stale stale
                        :scope-ns-prefixes scope}))))))
      :cljs nil)))

;; =============================================================================
;; Staleness Queries
;; =============================================================================

#?(:clj
   (defn stale-coverage
     "Returns a map of all functions with stale coverage in the given scope.

      For each stale function, returns:
      - :sealed-sig - the signature recorded when test was sealed
      - :current-sig - the current signature of the function
      - :tested-by - the test(s) covering this function

      Uses global config if no argument provided."
     ([] (stale-coverage (:scope-ns-prefixes @*config*)))
     ([scope-ns-prefixes]
      (if (empty? scope-ns-prefixes)
        {}
        (let [all-fns (gr.externs/all-in-scope-functions scope-ns-prefixes)
              covered-fns (filter cov/covered? all-fns)]
          (into {}
            (for [fn-sym covered-fns
                  :let [sealed (cov/sealed-signature fn-sym)
                        current (sig/signature fn-sym)]
                  :when (and sealed (not= sealed current))]
              [fn-sym {:sealed-sig sealed
                       :current-sig current
                       :tested-by (cov/covered-by fn-sym)}])))))))

#?(:clj
   (defn stale-functions
     "Returns a set of all functions with stale coverage in scope.

      Uses global config if no argument provided."
     ([] (stale-functions (:scope-ns-prefixes @*config*)))
     ([scope-ns-prefixes]
      (set (keys (stale-coverage scope-ns-prefixes))))))

;; =============================================================================
;; Reporting and Statistics
;; =============================================================================

(defn print-coverage-report
  "Print a human-readable coverage report for a function.

   fn-sym - Fully qualified symbol of the function to check
   opts - Optional map with :scope-ns-prefixes (uses global config if not provided)"
  ([fn-sym] (print-coverage-report fn-sym {}))
  ([fn-sym opts]
   #?(:clj
      (let [{:keys [function proof-complete? covered uncovered stale scope-ns-prefixes note]}
            (verify-transitive-coverage fn-sym opts)]
        (println "Coverage Report for:" function)
        (println "=====================================")
        (println "Scope:" (if (seq scope-ns-prefixes) scope-ns-prefixes "(not configured)"))
        (when note (println "Note:" note))
        (println "Proof complete:" (if proof-complete? "YES" "NO"))
        (when (seq covered)
          (println)
          (println "Covered:" (count covered))
          (doseq [f (sort-by str covered)]
            (let [is-stale (contains? stale f)]
              (println (if is-stale "  ~ (STALE)" "  +") f))))
        (when (seq uncovered)
          (println)
          (println "UNCOVERED:" (count uncovered))
          (doseq [f (sort-by str uncovered)]
            (println "  -" f)))
        (when (seq stale)
          (println)
          (println "STALE:" (count stale))
          (doseq [f (sort-by str stale)]
            (println "  ~" f "- needs re-sealing"))))
      :cljs
      (println "Coverage report not available in ClojureScript"))))

(defn uncovered-in-scope
  "Find all guardrailed functions in scope that lack test coverage.

   Uses global config if no argument provided."
  ([] (uncovered-in-scope (:scope-ns-prefixes @*config*)))
  ([scope-ns-prefixes]
   #?(:clj
      (if (empty? scope-ns-prefixes)
        #{}
        (let [all-fns (gr.externs/all-in-scope-functions scope-ns-prefixes)]
          (into #{} (remove cov/covered? all-fns))))
      :cljs #{})))

(defn coverage-stats
  "Returns coverage statistics for functions in the given namespace scope.

   Uses global config if no argument provided.

   Returns map with:
   - :total - total number of guardrailed functions in scope
   - :covered - number with declared coverage
   - :uncovered - number without coverage
   - :stale - number with stale coverage
   - :fresh - number with fresh (up-to-date) coverage
   - :coverage-pct - percentage covered (including stale)"
  ([] (coverage-stats (:scope-ns-prefixes @*config*)))
  ([scope-ns-prefixes]
   #?(:clj
      (if (empty? scope-ns-prefixes)
        {:total 0 :covered 0 :uncovered 0 :stale 0 :fresh 0 :coverage-pct 100.0
         :note "No scope configured"}
        (let [all-fns (gr.externs/all-in-scope-functions scope-ns-prefixes)
              total (count all-fns)
              covered-fns (filter cov/covered? all-fns)
              covered-count (count covered-fns)
              uncovered-count (- total covered-count)
              stale-count (count (filter stale? covered-fns))
              fresh-count (- covered-count stale-count)]
          {:total total
           :covered covered-count
           :uncovered uncovered-count
           :stale stale-count
           :fresh fresh-count
           :coverage-pct (if (zero? total)
                           100.0
                           (* 100.0 (/ covered-count total)))}))
      :cljs
      {:total 0 :covered 0 :uncovered 0 :stale 0 :fresh 0 :coverage-pct 100.0})))
