(ns fulcro-spec.proof
  "Tools for verifying transitive test coverage proofs.

   This namespace provides functions to verify that a function and all its
   transitive dependencies have declared test coverage. This enables building
   a provable chain of tests from low-level functions up to application logic.

   Configuration:
   Use `configure!` to set global options once for your project:
   - :scope-ns-prefixes - set of namespace prefix strings to include (e.g. #{\"myapp\"})
   - :enforce? - when true, when-mocking!! and provided!! will throw on missing coverage

   Example:
     (proof/configure! {:scope-ns-prefixes #{\"myapp\" \"myapp.lib\"}
                        :enforce? true})"
  (:require
    [fulcro-spec.coverage :as cov]
    [com.fulcrologic.guardrails.impl.externs :as gr.externs]
    #?(:clj [clojure.set :as set])))

;; =============================================================================
;; Global Configuration
;; =============================================================================

(defonce ^:dynamic *config*
  (atom {:scope-ns-prefixes #{}
         :enforce? false}))

(defn configure!
  "Configure global proof settings. Options:
   - :scope-ns-prefixes - Set of namespace prefix strings that define which
                          namespaces are 'in scope' for coverage checking.
                          E.g., #{\"myapp\"} includes myapp.core, myapp.db, etc.
   - :enforce? - When true, when-mocking!! and provided!! will throw exceptions
                 if mocked functions lack transitive coverage. When false, they
                 behave like when-mocking!/provided!. Default: false.

   Call this once at the start of your test suite or in a test fixture."
  [opts]
  (swap! *config* merge opts))

(defn get-config
  "Returns the current proof configuration."
  []
  @*config*)

(defn- effective-scope
  "Returns the scope-ns-prefixes to use, preferring explicit opts over global config."
  [opts]
  (or (:scope-ns-prefixes opts)
      (:scope-ns-prefixes @*config*)))

(defn- enforcement-enabled?
  "Returns whether proof enforcement is enabled."
  []
  (:enforce? @*config*))

;; =============================================================================
;; Core Verification Functions
;; =============================================================================

(defn verify-transitive-coverage
  "Verify that fn-sym and all its transitive dependencies have test coverage.

   Returns a report map:
   - :function - the function checked
   - :scope-ns-prefixes - the namespace scope used
   - :transitive-deps - all guardrailed functions in the call graph
   - :covered - set of functions with declared coverage
   - :uncovered - set of functions without declared coverage
   - :proof-complete? - true if all trackable deps are covered

   Options (optional if global config is set):
   - :scope-ns-prefixes - set of namespace prefix strings to include"
  ([fn-sym] (verify-transitive-coverage fn-sym {}))
  ([fn-sym opts]
   #?(:clj
      (let [scope (effective-scope opts)]
        (if (empty? scope)
          {:function fn-sym
           :scope-ns-prefixes scope
           :transitive-deps #{}
           :covered #{}
           :uncovered #{}
           :proof-complete? true
           :note "No scope configured - use configure! or pass :scope-ns-prefixes"}
          (let [all-deps (gr.externs/transitive-calls fn-sym scope)
                covered (into #{} (filter cov/covered? all-deps))
                uncovered (set/difference all-deps covered)]
            {:function fn-sym
             :scope-ns-prefixes scope
             :transitive-deps all-deps
             :covered covered
             :uncovered uncovered
             :proof-complete? (empty? uncovered)})))
      :cljs
      {:function fn-sym
       :scope-ns-prefixes (effective-scope opts)
       :transitive-deps #{}
       :covered #{}
       :uncovered #{}
       :proof-complete? true
       :note "Call graph analysis not available in ClojureScript"})))

(defn fully-tested?
  "Returns true if the function and all its transitive dependencies have
   declared test coverage within the configured scope.

   This is the simple query function for checking coverage status.
   Configure scope first with `configure!` or pass opts.

   Examples:
     (fully-tested? 'myapp.orders/create-order)
     (fully-tested? 'myapp.orders/create-order {:scope-ns-prefixes #{\"myapp\"}})"
  ([fn-sym] (fully-tested? fn-sym {}))
  ([fn-sym opts]
   (:proof-complete? (verify-transitive-coverage fn-sym opts))))

(defn why-not-tested?
  "Returns a set of uncovered functions if fn-sym is not fully tested,
   or nil if it is fully tested. Useful for understanding what's missing.

   Examples:
     (why-not-tested? 'myapp.orders/create-order)
     ;; => #{myapp.db/save! myapp.validation/check}"
  ([fn-sym] (why-not-tested? fn-sym {}))
  ([fn-sym opts]
   (let [{:keys [proof-complete? uncovered]} (verify-transitive-coverage fn-sym opts)]
     (when-not proof-complete?
       uncovered))))

;; =============================================================================
;; Assertion and Enforcement
;; =============================================================================

(defn assert-transitive-coverage!
  "Assert that fn-sym has complete transitive coverage.
   Throws an exception if any in-scope dependency lacks coverage.
   Used by when-mocking!! at test time.

   When called without opts, uses global config.
   Only throws if :enforce? is true in config (or if opts explicitly passed)."
  ([fn-sym] (assert-transitive-coverage! fn-sym {}))
  ([fn-sym opts]
   #?(:clj
      (let [scope (effective-scope opts)
            ;; If opts were explicitly passed with scope, always enforce
            ;; Otherwise respect the global :enforce? flag
            should-enforce? (or (seq (:scope-ns-prefixes opts))
                               (enforcement-enabled?))]
        (when (and should-enforce? (seq scope))
          (let [{:keys [uncovered proof-complete?]} (verify-transitive-coverage fn-sym opts)]
            (when-not proof-complete?
              (throw (ex-info (str "Incomplete transitive coverage for " fn-sym
                                ". Uncovered functions: " (pr-str uncovered))
                       {:function fn-sym
                        :uncovered uncovered
                        :scope-ns-prefixes scope}))))))
      :cljs nil)))

;; =============================================================================
;; Reporting and Statistics
;; =============================================================================

(defn print-coverage-report
  "Print a human-readable coverage report for a function.

   fn-sym - Fully qualified symbol of the function to check
   opts - Optional map with :scope-ns-prefixes (uses global config if not provided)"
  ([fn-sym] (print-coverage-report fn-sym {}))
  ([fn-sym opts]
   #?(:clj
      (let [{:keys [function proof-complete? covered uncovered scope-ns-prefixes note]}
            (verify-transitive-coverage fn-sym opts)]
        (println "Coverage Report for:" function)
        (println "=====================================")
        (println "Scope:" (if (seq scope-ns-prefixes) scope-ns-prefixes "(not configured)"))
        (when note (println "Note:" note))
        (println "Proof complete:" (if proof-complete? "YES" "NO"))
        (when (seq covered)
          (println)
          (println "Covered:" (count covered))
          (doseq [f (sort-by str covered)]
            (println "  +" f)))
        (when (seq uncovered)
          (println)
          (println "UNCOVERED:" (count uncovered))
          (doseq [f (sort-by str uncovered)]
            (println "  -" f))))
      :cljs
      (println "Coverage report not available in ClojureScript"))))

(defn uncovered-in-scope
  "Find all guardrailed functions in scope that lack test coverage.

   Uses global config if no argument provided."
  ([] (uncovered-in-scope (:scope-ns-prefixes @*config*)))
  ([scope-ns-prefixes]
   #?(:clj
      (if (empty? scope-ns-prefixes)
        #{}
        (let [all-fns (gr.externs/all-in-scope-functions scope-ns-prefixes)]
          (into #{} (remove cov/covered? all-fns))))
      :cljs #{})))

(defn coverage-stats
  "Returns coverage statistics for functions in the given namespace scope.

   Uses global config if no argument provided.

   Returns map with:
   - :total - total number of guardrailed functions in scope
   - :covered - number with declared coverage
   - :uncovered - number without coverage
   - :coverage-pct - percentage covered"
  ([] (coverage-stats (:scope-ns-prefixes @*config*)))
  ([scope-ns-prefixes]
   #?(:clj
      (if (empty? scope-ns-prefixes)
        {:total 0 :covered 0 :uncovered 0 :coverage-pct 100.0
         :note "No scope configured"}
        (let [all-fns (gr.externs/all-in-scope-functions scope-ns-prefixes)
              total (count all-fns)
              covered-count (count (filter cov/covered? all-fns))
              uncovered-count (- total covered-count)]
          {:total total
           :covered covered-count
           :uncovered uncovered-count
           :coverage-pct (if (zero? total)
                           100.0
                           (* 100.0 (/ covered-count total)))}))
      :cljs
      {:total 0 :covered 0 :uncovered 0 :coverage-pct 100.0})))
