(ns fulcro-spec.coverage
  "Registry for tracking which functions are covered by which tests.

   This namespace provides the foundation for transitive test coverage proofs.
   Tests declare which functions they cover via the :covers metadata in the
   specification macro, and this registry tracks those declarations.")

;; Registry: {qualified-fn-symbol -> #{qualified-test-symbols}}
(defonce coverage-registry (atom {}))

(defn register-coverage!
  "Register that `test-sym` covers `fn-sym`.
   Both should be fully qualified symbols."
  [test-sym fn-sym]
  (swap! coverage-registry update fn-sym (fnil conj #{}) test-sym))

(defn covered-by
  "Returns set of test symbols that cover the given function, or nil if none."
  [fn-sym]
  (get @coverage-registry fn-sym))

(defn covered?
  "Returns true if fn-sym has at least one test covering it."
  [fn-sym]
  (boolean (seq (covered-by fn-sym))))

(defn clear-registry!
  "Clear the coverage registry. Useful for testing."
  []
  (reset! coverage-registry {}))

(defn all-covered-functions
  "Returns all function symbols that have declared test coverage."
  []
  (keys @coverage-registry))

(defn coverage-summary
  "Returns a summary map of coverage information."
  []
  {:total-functions (count @coverage-registry)
   :coverage-map @coverage-registry})
