(ns franz.core
  (:require [clojure.data.avl :as avl]
            [tailrecursion.priority-map :as pm]
            [goog.async.nextTick]))

(defprotocol ILog
  (-read [this offset])
  (-read-all [this offset])
  (-dense [this])
  (-clear-to-offset [this offset])
  (-clear-to-size [this size]))

(deftype Log [log key-index write-point]
  ILog
  (-read [this offset]
    (avl/nearest log >= offset))
  (-read-all [this offset]
    (seq (avl/subrange log >= offset)))
  (-dense [this]
    (vals log))
  (-clear-to-offset [this offset]                            ;TODO maybe check if we actually resize first? Might save some redundant operations here.
    (Log. (avl/subrange log >= offset)
          (into (empty key-index)
                (filter (fn [[k v]]
                          (< v offset)))
                key-index)
          write-point))
  (-clear-to-size [this size]
    (let [pos (- (count log) size)]
      (if (pos? pos)
        (let [log* (second (avl/split-at pos log))
              retention-point* (nth log* 0)
              key-index* (into (empty key-index)
                               (remove (fn [[k v]] (< v retention-point*)))
                               key-index)]
          (Log. log*
                key-index*
                write-point))
        this)))
  ICollection
  (-conj [this [k v :as e]]
    (if e
      (Log.
        (cond-> log
                (and key (key-index k)) (dissoc (key-index k))
                :do (assoc write-point [k v]))
        (if k
          (assoc key-index k write-point)
          key-index)
        (inc write-point))
      (Log.
        log
        key-index
        (inc write-point))))
  IIndexed
  (-nth [this n]
    (get log n))
  (-nth [this n not-found]
    (get log n not-found))
  ISeqable
  (-seq [this]
    (map log (range write-point)))
  ICounted
  (-count [this]
    write-point))

(defn read
  "Reads the next message at or after the given offset.
  Returns a pair of the next offset and it's message."
  [log offset]
  (-read log offset))

(defn read-all
  "Reads all messages starting at or after the given offset.
  Returns a seq of pairs of the next offset and it's message."
  ([log]
   (-read-all log 0))
  ([log offset]
   (-read-all log offset)))

(defn dense
  "Returns the messages in the log."
  [log]
  (-dense log))

(defn clear-to-offset
  "Moves the retention point to the given offset,
  this drops all messages up to it, excluding."
  [log offset]
  (-clear-to-offset log offset))

(defn clear-to-size
  "Moves the retention point by dropping messages until the log has the given size."
  [log size]
  (-clear-to-size log size))

(defn log [& keyvals]
  (into (->Log (avl/sorted-map-by <) {} 0)
        (partition 2 keyvals)))

(defprotocol ITopic
  (-send! [this msg] [this key msg])
  (-subscribe! [this key position handler])
  (-unsubscribe! [this key])
  (-tick! [this] [this key])
  (-flush! [this] [this key])
  (-positions [this])
  (-schedule! [this])
  (-compact! [this]))

(deftype Topic [^:mutable log ^:mutable handlers ^:mutable handler-positions ^:mutable idle max-size]
  ITopic
  (-send!
    [this msg]
    (-send! this nil msg))
  (-send!
    [this key msg]
    (set! log (conj log [key msg]))
    (-compact! this)
    (-schedule! this)
    this)
  (-subscribe!
    [this key position handler]
    (set! handlers (assoc handlers key handler))
    (set! handler-positions (assoc handler-positions key position))
    (-compact! this)
    (-schedule! this)
    this)
  (-unsubscribe! [this key]
    (set! handlers (dissoc handlers key))
    (set! handler-positions (dissoc handler-positions key))
    this)
  (-tick! [this]
    (let [handler-key (ffirst handler-positions)]
      (-tick! this handler-key)))
  (-tick! [this key]
    (let [position (handler-positions key)
          handler (handlers key)]
      (when (< position (count log))
        (let [[msg-offset msg] (read log position)]
          (set! handler-positions (assoc handler-positions
                                    key
                                    (inc msg-offset)))
          (handler msg-offset msg)
          [key msg-offset]))))
  (-flush! [this]
    (while (-tick! this))
    (-compact! this))
  (-flush! [this key]
    (while (-tick! this key))
    (-compact! this))
  (-positions [this]
    handler-positions)
  (-schedule! [this]
    (if idle
      (do
        (set! idle false)
        (goog.async.nextTick
          (fn tick! []
            (if (-tick! this)
              (do
                (set! idle false)
                (goog.async.nextTick tick!))
              (set! idle true))
            (-compact! this)))
        true)
      false))
  (-compact! [this]
    (when max-size
      (let [[handler-key position] (first handler-positions)]
        (when (< max-size (count log))
          (if (< max-size (- (count log) position))
            (set! log (clear-to-offset log position))
            (set! log (clear-to-size log max-size))))))
    this)
  IDeref
  (-deref [this]
    log)
  IReset
  (-reset! [this new-log]
    (set! log new-log)
    (set! handler-positions (into (empty handler-positions)
                                  (map vector
                                       (keys handler-positions)
                                       (repeat 0))))
    this))

(defn send!
  "Writes the given message onto the internal log.
  If a key is provided the previously existing message under this key will be replaced."
  ([topic msg]
   (-send! topic msg))
  ([topic key msg]
   (-send! topic key msg)))

(defn subscribe!
  "Adds the given handler to the list of subscribers under the given key,
  consuming from the log at position."
  [topic key position handler]
  (-subscribe! topic key position handler))

(defn unsubscribe!
  "Removes the handler for the given key."
  [topic key]
  (-unsubscribe! topic key))

(defn flush!
  "Executes the subscription function for the given key exhaustively over the given log
   until they are all positioned at the latest offset.
   If no key is provided all handlers will be flushed."
  ([topic]
   (-flush! topic))
  ([topic key]
   (-flush! topic key)))

(defn positions
  "Returns a map containing the next processed position/offset for each subscription function."
  [topic]
  (-positions topic))

(defn topic
  "Creates a new topic that holds a log and manages asynchronous execution of subscribed handling functions.
  A custom posibly prefilled log can be provided. When ommited or nil an empty franz.core/log will be created.
  When a size is provided, it will be used as a soft limit to shrink the log to, while nil means no shrinking is done.
  Old messages will only be disgarded however when every registered handler has read them."
  ([]
   (topic nil nil))
  ([log]
    (topic log nil))
  ([log size]
   (->Topic
     (or log (franz.core/log))
     {}
     (pm/priority-map-by <)
     true
     size)))
